/*
 * drivers/net/usb/aerial_usbdrv.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/workqueue.h>
#include <linux/mii.h>
#include <linux/usb.h>
#ifdef AERIAL_HOTPLUG_FIRMWARE
#include <linux/firmware.h>
#endif /* AERIAL_HOTPLUG_FIRMWARE */
#include "aerial_log.h"
#include "aerial_device.h"
#include "aerial_wid.h"
#include "aerial_ioctl.h"
#include "aerial_fw.h"
#include "aerial_sysfs.h"
#include "aerial_wmm.h"


/* macro & define  */
#define RX_MAX_QUEUE_MEMORY (60 * 1518)
#define	RX_QLEN(dev) (((dev)->udev->speed == USB_SPEED_HIGH) ? \
			(RX_MAX_QUEUE_MEMORY/(dev)->rx_urb_size) : 4)
#define	TX_QLEN(dev) (((dev)->udev->speed == USB_SPEED_HIGH) ? \
			(RX_MAX_QUEUE_MEMORY/(dev)->hard_mtu) : 4)


// reawaken network queue this soon after stopping; else watchdog barks
#define TX_TIMEOUT_JIFFIES	(5*HZ)

// throttle rx/tx briefly after some faults, so khubd might disconnect()
// us (it polls at HZ/4 usually) before we report too many false errors.
#define THROTTLE_JIFFIES	(HZ/8)

// between wakeups
#define UNLINK_TIMEOUT_MS	3

#define WID_TIMEOUT		(1000)

#define WID_USB_RETRY(p)	{ 					\
	struct aerial_private	*pv = (struct aerial_private*)p;	\
	if (atomic_read(&pv->modstate)==AERIAL_MODSTATE_EXIST) 		\
		mod_timer (&pv->wid_retry, jiffies +msecs_to_jiffies(WID_TIMEOUT) ); \
	};
/* Power save &  Power Management */
#define SetPortFeature                  (0x2300 | USB_REQ_SET_FEATURE)
#define ClearPortFeature                (0x2300 | USB_REQ_CLEAR_FEATURE)
#define GetPortStatus                   (0xa300 | USB_REQ_GET_STATUS)
#define USB_PORT_FEAT_SUSPEND           2   /* L2 suspend */
#define USB_PORT_STAT_SUSPEND           (0x01 << USB_PORT_FEAT_SUSPEND)

#define USB_GET_STAT_CYC    			10                      /* 10 msec   */
#define USB_GET_STAT_TMO    			(200/USB_GET_STAT_CYC)  /* 200 msec */
#define USB_PM_WAIT(p)      			{							\
		struct aerial_private* pv = (struct aerial_private*)p;		\
		wait_for_completion_interruptible_timeout(					\
			&(pv->pmact.complete),									\
			msecs_to_jiffies(USB_GET_STAT_CYC) );					\
}

#ifdef AERIAL_HOTPLUG_FIRMWARE
static char *firmware = AERIAL_DEFAULT_FIRMWARE;
module_param (firmware, charp, S_IRUGO);
MODULE_PARM_DESC (firmware,
		  "firmware name. Default/empty:"AERIAL_DEFAULT_FIRMWARE"");
#endif /* AERIAL_HOTPLUG_FIRMWARE */

/* static function and variable */
unsigned int aerial_align_chunk_size(unsigned int size);

static char* aerial_wid_status(signed char code);
static int   aerial_wid_check(struct aerial_private *priv, struct aerial_packet *res );
void aerial_wid_complete (struct urb *urb);
static void aerial_do_task(struct work_struct *work);
static void aerial_wid_retry (unsigned long param);
static void aerial_task_init(struct aerial_private *priv);
static int aerial_rx_fixup(struct aerial_usbnet *dev, struct sk_buff *skb);
static struct sk_buff *aerial_tx_fixup(struct aerial_usbnet *dev, struct sk_buff *skb, gfp_t flags);
static void aerial_unbind(struct aerial_usbnet *dev, struct usb_interface *intf);
static void aerial_bind_clean(struct aerial_private *priv);
static int aerial_bind(struct aerial_usbnet *dev, struct usb_interface *intf);
static void aerial_ctl_complete(struct urb *urb);
static int aerial_port_status( struct aerial_private *priv, 
								unsigned short type,
								unsigned short val,
                        		unsigned short* psta );
static int aerial_pm_thread(void *thr);
static int aerial_last_init(struct aerial_usbnet *dev );

static void aerial_defer_bh(struct aerial_usbnet *dev, struct sk_buff *skb, struct sk_buff_head *list);
static void aerial_rx_submit (struct aerial_usbnet *dev, struct urb *urb, gfp_t flags);
static void aerial_rx_complete (struct urb *urb);
static void aerial_intr_complete (struct urb *urb);
static int  aerial_unlink_urbs (struct aerial_usbnet *dev, struct sk_buff_head *q);
static void aerial_kevent (struct work_struct *work);
static void aerial_tx_complete (struct urb *urb);
static int  aerial_usbnet_get_endpoints(struct aerial_usbnet *dev, struct usb_interface *intf);
static int aerial_usbnet_init_status (struct aerial_usbnet *dev, struct usb_interface *intf);
static void aerial_usbnet_skb_return (struct aerial_usbnet *dev, struct sk_buff *skb);
static struct net_device_stats* aerial_usbnet_get_stats (struct net_device *net);
static void aerial_usbnet_defer_kevent (struct aerial_usbnet *dev, int work);
static int  aerial_usbnet_stop (struct net_device *net);
static int  aerial_usbnet_open (struct net_device *net);
static void aerial_usbnet_tx_timeout (struct net_device *net);
static int  aerial_usbnet_start_xmit (struct sk_buff *skb, struct net_device *net);
static void aerial_usbnet_bh (unsigned long param);
static void aerial_usbnet_disconnect (struct usb_interface *intf);
static int  aerial_usbnet_probe (struct usb_interface *udev, const struct usb_device_id *prod);
static void aerial_usbnet_clean(struct aerial_usbnet *dev,
				struct usb_interface *intf);
static int aerial_tx_process(struct sk_buff *skb, struct aerial_usbnet *dev);
static wlan_wmm_ac_e aerial_wmm_nextqueue(struct aerial_thread *thread);

#if 0
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
#define AERIAL_MAC_BUF_SIZE	18
static int aerial_print_mac(char dest[AERIAL_MAC_BUF_SIZE],
			    const unsigned char addr[ETH_ALEN]);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,32)
#define AERIAL_MAC_BUF_SIZE	18
static int aerial_print_mac(char dest[AERIAL_MAC_BUF_SIZE],
			    const unsigned char addr[ETH_ALEN]);
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,32) */
#endif

struct usb_host_endpoint *tmp_int_in=NULL;
#ifdef AERIAL_HOTPLUG_FIRMWARE
static int aerial_usb_download_fw(struct usb_interface *intf,
				  const char *filename);
#endif /* AERIAL_HOTPLUG_FIRMWARE */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
#include <linux/usb/ch9.h>
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35) */


/* global function and variable */
/******************************************************************************
 * aerial_align_chunk_size -
 *
 *****************************************************************************/
/* Tx mode between  "fixed length mode" or "Variable length mode */
#define DS_VERSION (2) /* set (1) or (2) */

unsigned int
aerial_align_chunk_size(unsigned int size)
{
#if (DS_VERSION==1)
	return (((size / AERIAL_BU1805GU_DMA_SIZE) * AERIAL_BU1805GU_DMA_SIZE) +
		((size % AERIAL_BU1805GU_DMA_SIZE) ? AERIAL_BU1805GU_DMA_SIZE : 0) +
		AERIAL_TERM_SIZE );
#elif (DS_VERSION==2)
	return ( 1536 );
#else
	"ERROR" /* To make a compilation error happened */
#endif
}

/******************************************************************************
 * aerial_wid_status -
 *
 *****************************************************************************/
static char*
aerial_wid_status(signed char code)
{
	switch(code){
	case WID_STAT_WRN_FW_TYPE_MISMATCH:
		return WID_STAT_NAME_WRN_FW_TYPE_MISMATCH;
	case WID_STAT_ERR_PACKET_ERROR:
		return WID_STAT_NAME_ERR_PACKET_ERROR;
	case WID_STAT_ERR_INTERNAL_ERROR:
		return WID_STAT_NAME_ERR_INTERNAL_ERROR;
	case WID_STAT_ERR_FW_CHECKSUM_ERROR:
		return WID_STAT_NAME_ERR_FW_CHECKSUM_ERROR;
	case WID_STAT_ERR_ILLEGAL_FWID:
		return WID_STAT_NAME_ERR_ILLEGAL_FWID;
	case WID_STAT_ERR_DATA_ERROR:
		return WID_STAT_NAME_ERR_DATA_ERROR;
	case WID_STAT_ERR_FCS_ERROR:
		return WID_STAT_NAME_ERR_FCS_ERROR;
	case WID_STAT_ERR_MTYPE_INCORRECT:
		return WID_STAT_NAME_ERR_MTYPE_INCORRECT;
	case WID_STAT_ERR_WID_INCORRECT:
		return WID_STAT_NAME_ERR_WID_INCORRECT;
	case WID_STAT_ERR_MSG_LENGTH:
		return WID_STAT_NAME_ERR_MSG_LENGTH;
	case WID_STAT_ERR_ILLEGAL_SEQNO:
		return WID_STAT_NAME_ERR_ILLEGAL_SEQNO;
	case WID_STAT_ERR:
		return WID_STAT_NAME_ERR;
	case WID_STAT_OK:
		return WID_STAT_NAME_OK;
	default:
		break;
	}
	return WID_STAT_NAME_UNKNOWN;
}

/******************************************************************************
 * aerial_wid_check -
 *
 *****************************************************************************/
static int
aerial_wid_check(struct aerial_private *priv, struct aerial_packet *res )
{
	if (res->h.type == HTYPE_CONFIG_RES) {
		struct aerial_wid_frame *wframe = to_wid_frame(res);
		if( wframe->wid == WID_STATUS){
			if ((priv->fw_ready_check != 0) &&
			    ((signed char)wframe->val[0] == -11)) {
				priv->fw_not_ready = 1;
			}
			else {
				aer_info("WID_STATUS CODE=%d (%s)\n", 
					 (signed char)wframe->val[0],
					 aerial_wid_status( wframe->val[0] ) );
			}
		}
		else {
			aer_debug("check_wps_msg(usbBuffQue) called l.%d\n",
				 __LINE__);
			check_wps_msg(priv, res->message); /* usbBuffQue in windows */
		}
		if (res->m.type == CTYPE_RESPONSE) {
			memcpy(priv->wid_response,
			       res, res->h.len + 2);
		}
		if (res->m.type == CTYPE_INFO) {
			if (wframe->wid == WID_STATUS){
    			if (res->message[7] == 0x0) {
    				aer_info("disconnected!\n");
    				//netif_stop_queue(priv->netdev);
    				//netif_carrier_off(priv->netdev);
    			} else {
    				aer_info("connected!\n");
    				//netif_carrier_on(priv->netdev);
    				//netif_wake_queue(priv->netdev);
    			}
   			} else if (wframe->wid == WID_DEVICE_READY){
				priv->fw_not_ready = 0;
				atomic_set(&priv->state, AERIAL_STATE_READY);
				aer_info("device ready!\n");
   			} else if (wframe->wid == WID_WAKE_STATUS){
    			if (res->message[7] == 0x0) {
					atomic_set(&priv->suspend_state, AERIAL_PREPARE_SUSPENDED);
					complete(&priv->pmthr.complete);
    				aer_develop2("WID_WAKE_STATUS(sleep=%u)\n", jiffies);
    			} else {
					atomic_set(&priv->suspend_state, AERIAL_IDLE);
					atomic_set(&priv->resume_state, AERIAL_IDLE);
					complete(&priv->pmact.complete);
    				aer_develop2("WID_WAKE_STATUS(active=%u)\n", jiffies);
    			}
			}
		}
		if (res->m.type == CTYPE_NETWORK) {
			struct aerial_wid_frame *wframe = to_wid_frame(res);
    			aer_info("Network Info! [0x%04x]\n", wframe->wid);
			aer_dump( &(res->m.body[4]), (res->m.len-8) );
		}
	}
	else{
		aer_err("WID IN unknown packet recv.(type=%x)\n", res->h.type );
		return -1;
	}
	return (int)res->m.type;
}


/******************************************************************************
 * aerial_wid_complete -
 *
 *****************************************************************************/
void 
aerial_wid_complete (struct urb *urb)
{
	struct aerial_private 	*priv = (struct aerial_private*)urb->context;
	struct aerial_packet    *res  = priv->rxbuf;
	unsigned long		flags;

	spin_lock_irqsave (&priv->wid_submit_lock, flags);

	if( (urb->pipe&USB_ENDPOINT_DIR_MASK)==USB_DIR_IN ){
		aer_develop("WID BULK-IN completion.\n");

		if(!priv->wid_urb){
			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
			return;
		}
		priv->wid_urb = NULL;
		if (urb->status != 0) {
			aer_debug("It fails in WID BUKL IN.(status=%d)\n", urb->status );

			aer_develop2("aerial_wid_complete (%d)(r=%x, s=%x, f=%x, t=%d)\n", 
						urb->status,
						atomic_read(&priv->resume_state), 
						atomic_read(&priv->suspend_state),
						atomic_read(&priv->force_active),
						priv->probe_flags.do_task_quit );

			if (urb) {
				usb_free_urb (urb);
				urb = NULL;
			}
			
			if (priv->probe_flags.do_task_quit) {
				spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
				goto do_complete;
			}
			WID_USB_RETRY(priv);

			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
		}
		else{
			aer_debug("WID BULK-IN success! herdlen=%d, urblen=%d\n", res->h.len+2, urb->actual_length );
			aer_dump( (void*)res,  ((res->h.len+2)>16) ? 16 : (res->h.len+2) );

			if( aerial_wid_check( priv, res )==CTYPE_RESPONSE ){
				complete(&priv->wid_complete);
			}
			if (urb) {
				usb_free_urb (urb);
				urb = NULL;
			}
			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
			aerial_do_task((struct work_struct*)&priv->wid_queue);

		}
	} else {
		aer_debug("WID BULK-OUT completion.\n");
		complete(&priv->wid_out_complete);
		spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
	}
	return;

do_complete:

	aerial_do_task((struct work_struct*)&priv->wid_queue);

	return;
}

/******************************************************************************
 * aerial_do_task -
 *
 *****************************************************************************/
static void
aerial_do_task(struct work_struct *work)
{
	struct aerial_usbnet		*dev;
	struct aerial_data    	*data;
	struct aerial_private 	*priv;
	unsigned long		flags;
	struct urb		*urb = NULL;
	int			ret = 0;

	priv = container_of((struct delayed_work*)work, struct aerial_private, wid_queue );

	/* task of WID BULK-IN stop */
	if (priv->probe_flags.do_task_quit) {
	    if (atomic_read(&priv->power_state) == AERIAL_PWR_STATE_ACTIVE) {
		    /* completion for terminatrion of bind */
		    complete(&priv->wid_do_task_complete);
	    } 
	    return;
	}

	dev  = priv->usbnetDev;
	data = (struct aerial_data *)dev->data;

	aer_develop("function start.(aerial_do_task)\n");

	while(1){
	        /* WID receive */
		if (!(urb = usb_alloc_urb (0, GFP_ATOMIC))) {
			aer_err("It fails in allocation. (URB-IN)\n");
			break;
		}
		usb_fill_bulk_urb (urb, dev->udev, data->wid_bulk_in,
				   priv->rxbuf, sizeof(struct aerial_packet),
				   aerial_wid_complete, priv);

		spin_lock_irqsave (&priv->wid_submit_lock, flags);
		ret = usb_submit_urb (urb, GFP_ATOMIC);
		if(ret!=0){
			aer_err("It fails in submit.(URB-IN)(ret=%d)\n", ret );
			if (urb) {
				usb_free_urb (urb);
				urb = NULL;
			}
			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
			break;
		}
		priv->wid_urb = urb;
		spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
		return;
	}
	WID_USB_RETRY(priv);
	return;
}

/******************************************************************************
 * aerial_wid_retry -
 *
 *****************************************************************************/
static void 
aerial_wid_retry (unsigned long param)
{
	struct aerial_private *priv;
	priv = (struct aerial_private *)param;

	if (schedule_delayed_work(&priv->wid_queue, 
				  msecs_to_jiffies(0)) != 1) {
		aer_err("%s: schedule_delayed_work failed\n", __func__);
	}
	return;
}

/******************************************************************************
 * aerial_task_init -
 *
 *****************************************************************************/
static void
aerial_task_init(struct aerial_private *priv)
{
	INIT_DELAYED_WORK(&priv->wid_queue, aerial_do_task);
	priv->wid_retry.function = aerial_wid_retry;
	priv->wid_retry.data = (unsigned long) priv;
	init_timer (&priv->wid_retry);
	init_completion(&priv->wid_do_task_complete);
	return;
}

/******************************************************************************
 * aerial_rx_fixup -
 *
 *****************************************************************************/
static int 
aerial_rx_fixup(struct aerial_usbnet *dev, struct sk_buff *skb)
{
	struct aerial_packet_header  	packet_head;

#if 1
	aer_develop("function start.(aerial_rx_fixup)(len=%d)\n", skb->len );
	aer_devdump( (void*)skb->data, skb->len );
#else
	aer_develop2("function start.(aerial_rx_fixup)(len=%d)\n", skb->len );
	aer_devdump2( (void*)skb->data, (skb->len<32)?skb->len:32 );
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	skb_copy_from_linear_data( skb, &packet_head, sizeof(packet_head) );
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	skb_copy_bits(skb, 0, &packet_head, sizeof(packet_head));
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	skb_pull(skb, sizeof(packet_head) );

	// data only
	if(packet_head.type!=HTYPE_DATA_IN){
		aer_err("receive type is abnormal.(0x%x)\n", packet_head.type );
		return 0;
	}
	if (packet_head.len > ETH_FRAME_LEN) {
		aer_err("receive size is abnormal.(%d)\n", packet_head.len);
		return 0;
	}
	skb_trim(skb, packet_head.len);
	return 1;
}

/******************************************************************************
 * aerial_tx_fixup -
 *
 *****************************************************************************/
static struct sk_buff* 
aerial_tx_fixup(struct aerial_usbnet *dev, struct sk_buff *skb,
					gfp_t flags)
{
	struct aerial_packet_header  	packet_head;
	struct sk_buff 			*skb2;
	int            			size;
	struct aerial_data    		*data = (struct aerial_data *)dev->data;
	struct aerial_private 		*priv = data->priv;;

#if 1
	aer_develop("function start.(aerial_tx_fixup)(len=%d)\n", skb->len );
	aer_devdump( (void*)skb->data, skb->len );
#else
	aer_develop2("function start.(aerial_tx_fixup)(len=%d)\n", skb->len );
	aer_devdump2( (void*)skb->data, (skb->len<32)?skb->len:32 );
#endif

	if ( (atomic_read(&priv->power_state) == AERIAL_PWR_STATE_SLEEP) ||	
	     (atomic_read(&priv->state)==AERIAL_STATE_NO_FW) ){
		if (skb) {
			dev_kfree_skb_any(skb);
			skb = NULL;
		}
		return NULL;
	}

	size = aerial_align_chunk_size( skb->len+sizeof(packet_head) );
	skb2 = dev_alloc_skb( size );
	if(!skb2){
		aer_err("It fails in allocation. (skb)\n");
		return NULL;

	}
	skb2->dev = dev->net;
	skb_put(skb2, size );

	packet_head.len  = skb->len;
	packet_head.type = HTYPE_DATA_OUT;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	skb_copy_to_linear_data(skb2, &packet_head, sizeof(packet_head));
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	memcpy(skb2->data, &packet_head, sizeof(packet_head));
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	skb_copy_bits(skb, 0, &(skb2->data[2]), skb->len);

	if (skb) {
		dev_kfree_skb_any(skb);
		skb = NULL;
	}

	skb = skb2;
	return skb;
}


/******************************************************************************
 * aerial_tx_thread -
 *
 *****************************************************************************/
static int
aerial_tx_thread(void *thr)
{
	wait_queue_t wait;
	struct aerial_private 	*priv = NULL;
	struct sk_buff *skb;
	wlan_wmm_ac_e ac; 
	unsigned long		flags;
	
	
	struct aerial_thread *thread = thr;
	if (thr)
		priv = thread->priv;
	
	else {
		aer_err("%s(thr=NULL) failed(l.%d)\n", __func__, __LINE__);
		return -1;
	}
		
	if (unlikely(priv == NULL)) {
		aer_err("priv=NULL failed(l.%d)\n", __LINE__);
		return -1;
	}
	
	aerial_activate_thread(thread);
	
	init_waitqueue_entry(&wait, current);
	
	
	if (aerial_wmm_init(priv) < 0) {
		aer_info("aerial_wmm_init(priv=%p) failed\n", priv);
		return -1;
	}
	srandom32(jiffies);
	
	
	thread->flags = 0;
	
	for(;;) {
		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&thread->waitQ, &wait);
		schedule();
		remove_wait_queue(&thread->waitQ, &wait);
		set_current_state(TASK_RUNNING);
		if (kthread_should_stop()) {
			aer_debug("main-thread: break from main thread\n");
			break;
		}
		
		for (;;) {
			if ((ac = aerial_wmm_nextqueue(thread)) < 0) {
				aer_err("aerial_wmm_nextqueue(thread=%p) on %d\n", thread, ac);
				break;
			}
			
			if (ac == PND_Q) {
				aer_debug("%s():: Queues are empty!", __func__);
				break;
			}
			spin_lock_irqsave (&priv->wmm.wmm_skbq_lock, flags);
			skb = priv->wmm.txSkbQ[ac].skbuf.next;
			list_del((struct list_head *) priv->wmm.txSkbQ[ac].skbuf.next);
			spin_unlock_irqrestore(&priv->wmm.wmm_skbq_lock, flags);
			aerial_to_idle( priv );
			if( aerial_tx_process(skb, priv->usbnetDev) ){
				atomic_dec(&priv->force_active);
			}
		}
	}
	aerial_deactivate_thread(thread);
	return 0;
}

/******************************************************************************
 * aerial_wmm_compute_txprob -
 *
 *****************************************************************************/
/* This function computes Tx prob and Tx priority for each queue depending on*/
/* the EDCA parameters                                                       */
void aerial_wmm_compute_txprob(struct aerial_private *priv)
{
	struct aerial_wmm *wmm = &priv->wmm;
	edca_param_t *edca_params = wmm->edca_params;
	ACCESS_CATEGORIES_T *priority = wmm->edca_priority;
	unsigned int *txprob = wmm->edca_txprob;
	edca_state_t edca_cs[NUM_AC] = {{0}};
	unsigned char slot_assigned = 0;
	unsigned int i = 0;
	unsigned int j = 0;
	unsigned int ac = 0;
	unsigned int total_numtx = 0;
	unsigned int total_numtxop = 0;
	unsigned int txop_prob[NUM_AC] = {0};

	aer_develop("ComputeTxProb In");

	/* Initialize the Back-off counter */
	for(ac = 0; ac < NUM_AC; ac++)
	{
		edca_cs[ac].cw   = (1 << edca_params[ac].ecwmin);
		edca_cs[ac].bcnt = (random32() & (edca_cs[ac].cw - 1)) +
				edca_params[ac].aifsn;

/* This is the number of frames that can be txd in the TXOP. This is */
/* actually a complex problem depending upon the packet length & TX  */
/* Rate. A highly simplified approach is adopted here by using a     */
/* striaght forward Time to Pkts conversion factor. Some  Pkt  */
/* Length & TX-Rates are used for determining this Conv-Factor.      */
		edca_cs[ac].nftxp = max((unsigned int)1,
			edca_params[ac].txop/TXOP_TO_PACKET_CONV_FACTOR);
	}

	for(i = 0; i < NUM_SLOTS; i++)
	{
		slot_assigned = 0;
		for(ac = 0; ac < NUM_AC; ac++)
		{
			if(edca_cs[ac].bcnt == 0)
			{
				if(slot_assigned == 0)
				{
					slot_assigned = 1;
				/* Congratulations!! You have won a TXOP */
					edca_cs[ac].numtx += edca_cs[ac].nftxp;
					edca_cs[ac].ntxopwon++;
					edca_cs[ac].cw = (1 << edca_params[ac].ecwmin);
				}
				else
				{
/* Internal Collision. A higher priority queue has already won the TXOP  . */
/* The CW parameters of the LP are updated as if its an external collision */
/* Note that the relative priorities of the queues is fixed here to:       */
/* AC_VO > AC_VI > AC_BE > AC_BK                                           */
					edca_cs[ac].numcol++;

					edca_cs[ac].cw = 2 * edca_cs[ac].cw;
					if(edca_cs[ac].cw > 
					   (1 << edca_params[ac].ecwmax))
						edca_cs[ac].cw =
						 (1 << edca_params[ac].ecwmax);
				}
				edca_cs[ac].bcnt =
					(random32() & (edca_cs[ac].cw - 1)) +
					edca_params[ac].aifsn;
			}
			else
			{
				edca_cs[ac].bcnt--;
			}
		}
	}

	for(ac = 0; ac < NUM_AC; ac++)
	{
		total_numtx += edca_cs[ac].numtx;
		total_numtxop += edca_cs[ac].ntxopwon;
	}

	if(total_numtx == 0)
	{
		/* Exception1 */
		aer_err("Error::ComputeTxProb Exc2");
		return;
	}

	/* The TX-Probabilities for each of the Queues is computed. This can */
	/* be used for making the scheduling decisions.                      */
	for(ac = 0; ac < NUM_AC; ac++)
	{
		txprob[ac]  = (100 * edca_cs[ac].numtx) / total_numtx;
	}

/* The Priority of the queue is decided based on probability of winning a TXOP*/
	for(ac = 0; ac < NUM_AC; ac++)
	{
		txop_prob[ac]  = (100 * edca_cs[ac].ntxopwon) / total_numtxop;
		priority[ac] = ac;
		for(i = 0; i < ac; i++)
		{
			if(txop_prob[ac] > txop_prob[i])
				break;
		}
		if(i < ac)
		{
			for(j = ac; j > i; j--)
				priority[j] = priority[j-1];
				priority[i] = ac;
			}
		}

	aer_develop("ComputeTxProb Out");
	return;
}



/* This function updates the Tx Prob and Tx Priority calculated in the driver */
/* context structure.                                                         */
void aerial_wmm_update_txparams(struct aerial_private *priv)
{
	struct aerial_wmm *wmm = &priv->wmm;
	ACCESS_CATEGORIES_T *priority  = wmm->edca_priority;
	unsigned int             *txprob    = wmm->edca_txprob;
	wlan_wmm_ac_e             QueueIndex = (AERIAL_MAX_AC_QUEUES - 2);
	unsigned int             ac         = 0;
	unsigned long		flags;

	spin_lock_irqsave (&priv->wmm.wmm_skbq_lock, flags);

	wmm->QScheduleProb[AC_BK_Q] = txprob[AC_BK];
	wmm->QScheduleProb[AC_BE_Q] = txprob[AC_BE];
	wmm->QScheduleProb[AC_VI_Q] = txprob[AC_VI];
	wmm->QScheduleProb[AC_VO_Q] = txprob[AC_VO];

	for(ac = 0; ac < NUM_AC; ac++)
	{
		wlan_wmm_ac_e QueueType;

		if(priority[ac] == AC_VO)
			QueueType = AC_VO_Q;
		else if(priority[ac] == AC_VI)
			QueueType = AC_VI_Q;
		else if(priority[ac] == AC_BE)
			QueueType = AC_BE_Q;
		else if(priority[ac] == AC_BK)
			QueueType = AC_BK_Q;
        else {
		aer_err("%s : Invalid queue type.(%d)\n", priority[ac]);
		QueueType = AC_BK_Q;
        }

		wmm->TxQPriority[QueueIndex] = QueueType;
		QueueIndex--;
	}

	spin_unlock_irqrestore (&priv->wmm.wmm_skbq_lock, flags);
}


/******************************************************************************
 * aerial_wmm_nextqueue -
 *
 *****************************************************************************/
/* This function selects the NextQ for Tx in Round Robin manner starting from */
/* highest priority AC. Till a high priority queue is becomes empty lower     */
/* priority queues are not scheduled.                                         */
static wlan_wmm_ac_e aerial_wmm_nextqueue(struct aerial_thread *thread)
{


	struct aerial_private 	*priv = NULL;
	struct aerial_wmm *wmm = NULL;
	wlan_wmm_ac_e QueueType    = PND_Q;
	unsigned int QueueIndex   = 0;
	wlan_wmm_ac_e *TxQPriority = NULL;
	unsigned long		flags;
	
	if (thread)
		priv = thread->priv;
	else {
		aer_err("%s(thread=NULL) failed(l.%d)\n", __func__, __LINE__);
		return -1;
	}
	if (unlikely(priv == NULL)) {
		aer_err("priv=NULL failed(l.%d)\n", __LINE__);
		return -1;
	}
	
	wmm = &priv->wmm;
	TxQPriority = wmm->TxQPriority;

	/* Currently the Queues are scheduled in a Round Robin manner */
	for(QueueIndex =
		    (AERIAL_MAX_AC_QUEUES - 2); QueueIndex > 0 ; QueueIndex--)
	{
		spin_lock_irqsave (&priv->wmm.wmm_skbq_lock, flags);
		QueueType = TxQPriority[QueueIndex];
		if(!list_empty((struct list_head *)
			       &priv->wmm.txSkbQ[QueueType].skbuf)) {
			spin_unlock_irqrestore (&priv->wmm.wmm_skbq_lock,flags);
			return QueueType;
		}
		spin_unlock_irqrestore (&priv->wmm.wmm_skbq_lock, flags);
	}
	return PND_Q;
}

/******************************************************************************
 * aerial_unbind -
 *
 *****************************************************************************/
static void 
aerial_unbind(struct aerial_usbnet *dev, struct usb_interface *intf)
{
	unsigned long		flags;
	struct aerial_data    	*data = (struct aerial_data *)dev->data;
	struct aerial_private 	*priv = data->priv;
	int			i;
	int ret;
	aer_develop("function start.(aerial_unbind)\n");
	
	/* set flag for information of quit */
	priv->probe_flags.do_task_quit = 1;
	atomic_set(&priv->modstate, AERIAL_MODSTATE_NONE);


	/* WID BULK_IN is running */
	for (i = 0; i < AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N; i++) {
		spin_lock_irqsave (&priv->wid_submit_lock, flags);
		if (priv->wid_urb == NULL) {
			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
			break;
		}
		ret = usb_unlink_urb(priv->wid_urb);
		spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
		/* usb_unlink_urb() is success */
		if (ret == -EINPROGRESS)
			break;
		
		/* failure */
		aer_info("WID BULK_IN : usb_unlink_urb failed on %d\n", ret);
		msleep(100);
	}
	if (i >= AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N) {
		aer_err("WID BULK_IN : usb_unlink_urb failed %d times\n",
			AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N);
	}

	/* wake up tx_thread */
	wake_up_interruptible(&priv->txthr.waitQ);
	aerial_terminate_thread(&priv->txthr);

	aerial_terminate_thread(&priv->pmthr);
	aerial_terminate_thread(&priv->pmact);
	
	if (atomic_read(&priv->power_state) == AERIAL_PWR_STATE_ACTIVE) {
		/*  wait completion */
		if ((ret =
		     wait_for_completion_interruptible(
			     &priv->wid_do_task_complete))) {
			aer_err("%s: "
				"wait_for_completion_interruptible"
				"(task_complete) failed on %d\n",
				__func__, ret);
		}
	}

	if (priv->txthr.pid) {
		if ((ret =
		     wait_for_completion_interruptible(&priv->txthr.complete)))
			{
				aer_err("%s: wait_for_completion_interruptible"
					"(tx_thr complete) failed on %d\n",
					__func__, ret);
			}
	}

	/* terminate */
	/* timer */
	del_timer (&priv->wid_retry);

	/* wid_queue */
	(void)cancel_delayed_work(&priv->wid_queue);
	/* we don't hold rtnl here ... */
	flush_scheduled_work();

	aerial_bind_clean(priv);
	return;
}

static void
aerial_bind_clean(struct aerial_private *priv)
{

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
	int	i;
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

	if (priv == NULL)
		return;

	/* free fw_imagge */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	if(priv->fw_image) {
		kfree(priv->fw_image);
		priv->fw_image = NULL;
	}
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	for(i = 0; i < FIRMWARE_N_BLOCKS; i++) {
		if(priv->fw_image[i]) {
			kfree(priv->fw_image[i]);
			priv->fw_image[i] = NULL;
		}
	}
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	/* free priv->rxbuf */
	if (priv->rxbuf) {
		kfree(priv->rxbuf);
		priv->rxbuf = NULL;
	}

	/* free priv->wid_request */
	if (priv->wid_request) {
		kfree(priv->wid_request);
		priv->wid_request = NULL;
	}

	/* free priv->wid_response */
	if (priv->wid_response) {
		kfree(priv->wid_response);
		priv->wid_response = NULL;
	}

	/* free priv */
	if(priv) {
		kfree(priv);
		priv = NULL;
	}
	return;
}

/******************************************************************************
 * aerial_bind -
 *
 *****************************************************************************/
static int 
aerial_bind(struct aerial_usbnet *dev, struct usb_interface *intf)
{
	struct aerial_data    	*data;
	struct aerial_private 	*priv = NULL;
	int status;

	int			i;

	aer_develop("function start.(aerial_bind)\n");

	data = (struct aerial_data *)dev->data;
	memset(data, 0x00, sizeof(struct aerial_data));

	if ((data->priv = kzalloc(sizeof(struct aerial_private),
				  GFP_KERNEL)) == NULL) {
		aer_err("It fails in allocation. (private)\n");
		return -ENOMEM;
	}

	if ((data->priv->wid_response = 
	     kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer wid_response\n");
		status = -ENOMEM;
		goto out;
	}

	if ((data->priv->wid_request = 
	     kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer wid_request\n");
		status = -ENOMEM;
		goto out1;
	}

	if ((data->priv->rxbuf =
	     kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer rxbuf\n");
		status = -ENOMEM;
		goto out2;
	}

	dev->status = tmp_int_in;

	priv =data->priv;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	if ((priv->fw_image = kzalloc(FIRMWARE_SIZE_MAX, GFP_KERNEL)) == NULL) {
		aer_err("It fails in allocation. (firmware)\n");
		status = -ENOMEM;
		goto out3;

	}
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	for(i = 0; i < FIRMWARE_N_BLOCKS; i++) {
		if ((priv->fw_image[i] = kzalloc(FIRMWARE_BLOCK_MAX_SIZE,
						 GFP_KERNEL)) == NULL) {
			aer_err("It fails in allocation. (firmware %d/%d)\n",
				i, FIRMWARE_N_BLOCKS);
			status = -ENOMEM;
			if (i > 0) {
				for (; i >= 0; i--) {
					kfree(priv->fw_image[i]);
				}
			}
			goto out3;
		}
	}
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	priv->netdev    = dev->net;
	priv->usbnetDev = dev;
	priv->fw_size   = 0;


    /* Initialize Default WMM params */
	priv->wmm.QScheduleProb[PND_Q]   = TXOP_FOR_PND_Q   ;
	priv->wmm.QScheduleProb[AC_BK_Q] = TXOP_FOR_AC_BK_Q ;
	priv->wmm.QScheduleProb[AC_BE_Q] = TXOP_FOR_AC_BE_Q ;
	priv->wmm.QScheduleProb[AC_VI_Q] = TXOP_FOR_AC_VI_Q ;
	priv->wmm.QScheduleProb[AC_VO_Q] = TXOP_FOR_AC_VO_Q ;
	priv->wmm.QScheduleProb[HIP_Q]   = TXOP_FOR_HIP_Q   ;

	for(i = 0; i < AERIAL_MAX_AC_QUEUES; i++) {
		/* Assign the Tx Priority to each queue */
		priv->wmm.TxQPriority[i]  = i;
	}

	/* Initialize Default EDCA Parameters */
	priv->wmm.edca_params[AC_VO].aifsn  = 2;
	priv->wmm.edca_params[AC_VO].ecwmin = 2;
	priv->wmm.edca_params[AC_VO].ecwmax = 3;
	priv->wmm.edca_params[AC_VO].txop   = 1504;

	priv->wmm.edca_params[AC_VI].aifsn  = 2;
	priv->wmm.edca_params[AC_VI].ecwmin = 3;
	priv->wmm.edca_params[AC_VI].ecwmax = 4;
	priv->wmm.edca_params[AC_VI].txop   = 3008;

	priv->wmm.edca_params[AC_BE].aifsn  = 3;
	priv->wmm.edca_params[AC_BE].ecwmin = 4;
	priv->wmm.edca_params[AC_BE].ecwmax = 10;
	priv->wmm.edca_params[AC_BE].txop   = 0;

	priv->wmm.edca_params[AC_BK].aifsn  = 7;
	priv->wmm.edca_params[AC_BK].ecwmin = 4;
	priv->wmm.edca_params[AC_BK].ecwmax = 10;
	priv->wmm.edca_params[AC_BK].txop   = 0;


	atomic_set(&priv->state, AERIAL_STATE_NO_FW);
	atomic_set(&priv->power_state, AERIAL_PWR_STATE_ACTIVE);
	atomic_set(&priv->power_mngt,  AERIAL_PWR_MNGT_ACTIVE);
	atomic_set(&priv->suspend_state, AERIAL_IDLE);
	atomic_set(&priv->resume_state, AERIAL_IDLE);
	atomic_set(&priv->net_suspend_state, AERIAL_IDLE);
	atomic_set(&priv->force_active, 0);
	spin_lock_init (&priv->pm_lock);
	atomic_set(&priv->modstate, AERIAL_MODSTATE_NONE);
	sema_init(&priv->wid_req_lock, 1);
	init_completion(&priv->wid_complete);
	init_completion(&priv->wid_out_complete);
	aerial_task_init(priv);

	spin_lock_init (&priv->wid_submit_lock);

#ifdef AERIAL_HOTPLUG_FIRMWARE
	sema_init(&priv->fw_lock, 1);

	if ((firmware == NULL) || (firmware[0] == 0))
		priv->firmware_file = 0;
	else 
		priv->firmware_file = firmware;

#endif /* AERIAL_HOTPLUG_FIRMWARE */

        aerial_wireless_attach(priv->netdev);
	if((status = aerial_usbnet_get_endpoints(dev,intf)) < 0 ){
		aer_err("unknown endpoint.\n");
		goto out4;
	}

	if( ((data->wid_bulk_in>>15)&0x0f)!=AERIAL_EP_WIDIN    ||
            ((data->wid_bulk_out>>15)&0x0f)!=AERIAL_EP_WIDOUT  ||
	    ((dev->in>>15)&0x0f)!=AERIAL_EP_DATIN              ||
	    ((dev->out>>15)&0x0f)!=AERIAL_EP_DATOUT ){
		aer_err("unknown endpoint number.\n");
		status = -1;
		goto out4;
	}

	priv->txthr.priv = priv;
	init_waitqueue_head(&priv->txthr.waitQ);
	init_completion(&priv->txthr.complete);
	aerial_create_thread(aerial_tx_thread, &priv->txthr, "aerial_txthr");
	if (schedule_delayed_work(&priv->wid_queue,
				  msecs_to_jiffies(0)) != 1) {
		aer_err("%s: schedule_delayed_work failed\n", __func__);
		return -ESRCH;
	}
	priv->probe_flags.do_task_quit = 0;

	priv->pmthr.priv = priv;
	priv->pmact.priv = priv;
	init_completion(&priv->pmthr.complete);
	init_completion(&priv->pmact.complete);
	aerial_create_thread(aerial_pm_thread, &priv->pmthr, "aerial_pmthr");
	return 0;


out4:
out3:
out2:
out1:
out:
	aerial_bind_clean(priv);

	return status;
}

/******************************************************************************
 * aerial_get_port_status  -
 *
 *****************************************************************************/
static void 
aerial_ctl_complete(struct urb *urb)
{
	struct aerial_private 	*priv = (struct aerial_private*)urb->context;
	urb->dev->parent = priv->parent;
	usb_free_urb(urb);
	return;
}

/******************************************************************************
 * aerial_get_port_status  -
 *
 *****************************************************************************/
static int
aerial_port_status( struct aerial_private *priv, 
						unsigned short type,
						unsigned short val,
                        unsigned short* psta )
{
	int							ret;
    struct usb_ctrlrequest* 	ctrReq;
//	unsigned long				flags;
	struct urb					*urb = NULL;
	struct aerial_usbnet* 		dev = priv->usbnetDev;

	ctrReq = (struct usb_ctrlrequest*)&priv->port_status;

	if (!(urb = usb_alloc_urb (0, GFP_ATOMIC))) {
		aer_err("It fails in allocation. (URB-CONTROL)\n");
		return -1;
	}
	
//	spin_lock_irqsave (&priv->pm_lock, flags);
	ctrReq->bRequestType	= (type >> 8) & 0xFF;
	ctrReq->bRequest		= type & 0xFF;
 	ctrReq->wValue			= val;
	ctrReq->wIndex			= 1;
	ctrReq->wLength			= 2;
	
	usb_fill_control_urb(urb, dev->udev,
			 			 usb_sndctrlpipe(dev->udev, 0), 
						(char *) (void*)ctrReq,
						(char *) (void*)ctrReq,
						sizeof(unsigned short), 
						aerial_ctl_complete, priv );
	
	priv->parent = urb->dev->parent;
	urb->dev->parent = 0;

	ret = usb_submit_urb (urb, GFP_ATOMIC);
//	spin_unlock_irqrestore (&priv->pm_lock, flags);
	if( ret!=0 ) {
		aer_err("usb_submit_urb(URB-CONTROL) error. (0x%x,%d)\n", type, ret );
		return -1;
	}
    if(psta)
    	*psta = priv->port_status;
	return 0;
}

/******************************************************************************
 * aerial_to_idle  -
 *
 *****************************************************************************/
void
aerial_to_idle( struct aerial_private *priv )
{
	int		loop;

	atomic_inc(&priv->force_active);

	if( atomic_read(&priv->power_mngt)==AERIAL_PWR_MNGT_ACTIVE )
    	return;

RESUME_RETRY:

	for(loop=0; loop<USB_GET_STAT_TMO; loop++){
		if( atomic_read(&priv->resume_state)==AERIAL_IDLE ) 
			break;
		USB_PM_WAIT(priv);
	}
	if( atomic_read(&priv->resume_state)!=AERIAL_IDLE ) {
		aer_develop2("idle wait1.(r=%x, s=%x, f=%x)\n", 
					atomic_read(&priv->resume_state), 
					atomic_read(&priv->suspend_state),
					atomic_read(&priv->force_active) );
		goto WID_IN_RESTART;
	}


	switch( atomic_read(&priv->suspend_state) ) {
		case AERIAL_IDLE:
		case AERIAL_PREPARE_SUSPENDED:
			break;

		case AERIAL_EXECUTE_SUSPENDED:
			while( atomic_read(&priv->suspend_state)!=AERIAL_SUSPENDED ){
				aer_debug("idle wait2.(r=%x, s=%x, f=%x)\n", 
							atomic_read(&priv->resume_state), 
							atomic_read(&priv->suspend_state),
							atomic_read(&priv->force_active) );
				USB_PM_WAIT(priv);
			}
			break;

		case AERIAL_SUSPENDED:
			atomic_set(&priv->resume_state, AERIAL_PREPARE_RESUME );
			msleep(1);
			aerial_resume(priv, AERIAL_ON );
			break;
	}

	for(loop=0; loop<USB_GET_STAT_TMO; loop++){
		if( atomic_read(&priv->suspend_state)==AERIAL_IDLE || 
			atomic_read(&priv->suspend_state)==AERIAL_PREPARE_SUSPENDED )
			break;
		USB_PM_WAIT(priv);
	}

	if( atomic_read(&priv->suspend_state)!=AERIAL_IDLE && 
		atomic_read(&priv->suspend_state)!=AERIAL_PREPARE_SUSPENDED ){
		aer_develop2("idle retry.(r=%x, s=%x, f=%x)\n", 
					atomic_read(&priv->resume_state), 
					atomic_read(&priv->suspend_state),
					atomic_read(&priv->force_active) );
		goto WID_IN_RESTART;
	}
	return;

WID_IN_RESTART:
	mod_timer (&priv->wid_retry, jiffies +msecs_to_jiffies(0) );
	goto RESUME_RETRY;
}


/******************************************************************************
 * aerial_suspend  -
 *
 *****************************************************************************/
int
aerial_suspend( struct aerial_private *priv )
{
	int						i;
	unsigned long			flags;
	int						wait;
	unsigned short 			status = 0;
	struct aerial_usbnet* 	dev = priv->usbnetDev;
	int						ret = 0;

	if (netif_running (dev->net)) {
		atomic_set(&priv->net_suspend_state, AERIAL_SUSPENDED);
		ret = aerial_usbnet_stop(dev->net);
		if( ret != 0 ){
			aer_err("Aerial suspend :USB/Network stop error\n");
			return -1;
		}
	}

	priv->probe_flags.do_task_quit = 1;

	/* WID BULK_IN is running */
	for (i = 0; i < AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N; i++) {
		spin_lock_irqsave (&priv->wid_submit_lock, flags);
		if (priv->wid_urb == NULL) {
			spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
			break;
		}
		ret = usb_unlink_urb(priv->wid_urb);
		spin_unlock_irqrestore (&priv->wid_submit_lock, flags);
		/* usb_unlink_urb() is success */
		if (ret == -EINPROGRESS)
			break;
		
		/* failure */
		aer_info("Aerial suspend :usb_unlink_urb failed on %d\n", ret);
		msleep(100);
	}
	if (i >= AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N) {
		aer_err("Aerial suspend :usb_unlink_urb failed %d times\n",
		AERIAL_WID_BUILK_CANCEL_REQ_RETRY_N);
		return -1;
	}

SUSPEND_SET:

	if( aerial_port_status( priv, 
							SetPortFeature,
							USB_PORT_FEAT_SUSPEND,
                        	NULL ) != 0 ){
		aer_err("Aerial suspend : setting port stats error.\n");
		return -1;
	}

	for( wait=0; wait<USB_GET_STAT_TMO; wait++ ){
		if( aerial_port_status( priv, 
								GetPortStatus,
								0,
	                        	&status ) != 0 ){
			aer_err("Aerial suspend : get port stats error.\n");
			return -1;
		}

		if( (status & USB_PORT_STAT_SUSPEND) 	&&
			dev->rxq.qlen==0					&&
			dev->txq.qlen==0					)
			break;
		USB_PM_WAIT(priv);
	}
	if( wait >= USB_GET_STAT_TMO ) {
		aer_err("Aerial suspend : port stats error.(stat=0x%x)\n", status );
		USB_PM_WAIT(priv);
		goto SUSPEND_SET;
	}

	atomic_set(&priv->suspend_state, AERIAL_SUSPENDED);
	return 0;
}

/******************************************************************************
 * aerial_resume  -
 *
 *****************************************************************************/
int
aerial_resume( struct aerial_private *priv, int flag )
{
	int						wait;
	struct aerial_usbnet* 	dev = priv->usbnetDev;
	int						ret = 0;
	unsigned short 			status = 0;

	if( flag == AERIAL_ON ){

		if( aerial_port_status( priv, 
								ClearPortFeature,
								USB_PORT_FEAT_SUSPEND,
	                        	NULL ) != 0 ){
			aer_err("Aerial resume : clear port stats error.\n");
			return -1;
		}

		for( wait=0; wait<USB_GET_STAT_TMO; wait++ ){
 			USB_PM_WAIT(priv);
			if( aerial_port_status( priv, 
									GetPortStatus,
									0,
		                        	&status ) != 0 ){
				aer_err("Aerial resume : get port stats error.\n");
				return -1;
			}
			if( (status & USB_PORT_STAT_SUSPEND) == 0 )
				break;
		}
		if( wait > USB_GET_STAT_TMO ) {
			aer_err("Aerial resume : port stats error.(stat=0x%x)\n", status );
			return -1;
		}
	}

	priv->probe_flags.do_task_quit = 0;
	mod_timer (&priv->wid_retry, jiffies +msecs_to_jiffies(0) );
	if( atomic_read(&priv->net_suspend_state) ) {
		ret = aerial_usbnet_open(dev->net);
		if( ret != 0 ){
			aer_err("Aerial resume :USB/Network start error\n");
		}
		atomic_set(&priv->net_suspend_state, AERIAL_IDLE);
	}
	return ret;
}

/******************************************************************************
 * aerial_pm_thread -
 *
 *****************************************************************************/
static int
aerial_pm_thread(void *thr)
{
	int 					timeout;
	struct aerial_private 	*priv   = NULL;
	struct aerial_thread 	*thread = thr;
	unsigned short 			status = 0;
	struct aerial_usbnet* 	dev;

	if (thr)
		priv = thread->priv;
	else {
		aer_err("%s(thr=NULL) failed(l.%d)\n", __func__, __LINE__);
		return -1;
	}
	dev = priv->usbnetDev;
		
	if (unlikely(priv == NULL)) {
		aer_err("priv=NULL failed(l.%d)\n", __LINE__);
		return -1;
	}
	
	aerial_activate_thread(thread);

	do {
		timeout = wait_for_completion_interruptible_timeout(
				&priv->pmthr.complete,
				msecs_to_jiffies(1));

		aer_debug("aerial_pm_thread.(r=%x, s=%x, f=%x)\n", 
					atomic_read(&priv->resume_state), 
					atomic_read(&priv->suspend_state),
					atomic_read(&priv->force_active) );

		set_current_state(TASK_RUNNING);
		switch( atomic_read(&priv->suspend_state) ){
			case AERIAL_IDLE:
				break;
			case AERIAL_PREPARE_SUSPENDED:
				if( atomic_read(&priv->force_active)==AERIAL_OFF ){
					atomic_set(&priv->suspend_state, AERIAL_EXECUTE_SUSPENDED);
					if( aerial_suspend( priv ) != 0 ){
						atomic_set(&priv->suspend_state, AERIAL_PREPARE_SUSPENDED);
						aer_err("Aerial power management : suspend error\n");
						continue;
					}
					aer_develop2("sleep!(%u) \n", jiffies );
				}
				break;

			case AERIAL_SUSPENDED:
				if( atomic_read(&priv->resume_state) == AERIAL_IDLE) {
					if( aerial_port_status( priv, 
											GetPortStatus,
											0,
				                        	&status ) == 0 ){
						if( (status & USB_PORT_STAT_SUSPEND)== 0 ){
							atomic_set(&priv->resume_state, AERIAL_PREPARE_WAKEUP );
							aerial_resume(priv, AERIAL_OFF);
			   				aer_develop2("active!(%u)\n", jiffies);
						}
					}
					else
						aer_err("Aerial power management : get port stats error.\n");
				}
				break;
		}
		set_current_state(TASK_INTERRUPTIBLE);
	} while (!kthread_should_stop());
	return 0;
}

/******************************************************************************
 * aerial_last_init -
 *
 *****************************************************************************/
static int 
aerial_last_init(struct aerial_usbnet *dev )
{
	struct aerial_data    	*data;
	struct aerial_private 	*priv;
	int 			ret;
	struct usb_device	*xdev;
	char			*speed[5]={"UNKNOWN", "LOW", "FULL", "HIGH", "VARIABLE" };
	data = (struct aerial_data *)dev->data;
	priv =data->priv;

	aer_develop("function start.(aerial_last_init)\n");

	atomic_set(&priv->modstate, AERIAL_MODSTATE_EXIST);
	xdev = interface_to_usbdev (dev->intf);
	priv->maxpacket = usb_maxpacket (dev->udev, dev->out, 1);
	priv->speed     = xdev->speed;

	aer_info("Driver Version %s %s (%s %s)\n", AERIAL_VERSION,
#ifdef AERIAL_HOTPLUG_FIRMWARE
		 "hotplug",
#else /* AERIAL_HOTPLUG_FIRMWARE */
		 "normal",
#endif /* AERIAL_HOTPLUG_FIRMWARE */
		 __DATE__, __TIME__);

	aer_info("    Endpoint   : DATA-BULK-OUT=%d, WID-BULK-OUT=%d, DATA-BULK-IN=%d, WID-BULK-IN=%d\n",
			(dev->out>>15)&0x0f, (data->wid_bulk_out>>15)&0x0f, 
			(dev->in>>15)&0x0f,  (data->wid_bulk_in>>15)&0x0f );
	if(dev->status){
		unsigned	epnum;
		epnum = usb_rcvintpipe (dev->udev,
				dev->status->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
		aer_info("                 INT-IN=%d\n", (epnum>>15)&0x0f );
	}
	else{
	}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35)
	aer_info("    Bus Speed  : %s\n", (priv->speed>USB_SPEED_WIRELESS) ? 
                                         speed[USB_SPEED_UNKNOWN] : speed[priv->speed] ); 
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35) */
	aer_info("    Bus Speed  : %s\n", (priv->speed>USB_SPEED_VARIABLE) ? 
                                         speed[USB_SPEED_UNKNOWN] : speed[priv->speed] ); 
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,35) */
	aer_info("    Max Packet : %d\n", priv->maxpacket );

	ret = aerial_sysfs_init(priv);
	if (ret) {
		aer_err("failed register sysfs-entry\n");
		return -1;
	}
	return 0;
}


/******************************************************************************
 * aerial_defer_bh -
 *
 *****************************************************************************/
static void 
aerial_defer_bh(struct aerial_usbnet *dev, struct sk_buff *skb, struct sk_buff_head *list)
{
	unsigned long		flags;

	aer_develop("function start.(aerial_defer_bh)\n");

	spin_lock_irqsave(&list->lock, flags);
	__skb_unlink(skb, list);

	spin_unlock_irqrestore(&list->lock, flags);
	spin_lock_irqsave(&dev->done.lock, flags);

	__skb_queue_tail(&dev->done, skb);
	if (dev->done.qlen == 1)
		tasklet_schedule(&dev->bh);
	spin_unlock_irqrestore(&dev->done.lock, flags);
}


/******************************************************************************
 * aerial_rx_submit -
 *
 *****************************************************************************/
static void 
aerial_rx_submit (struct aerial_usbnet *dev, struct urb *urb, gfp_t flags)
{
	struct sk_buff		*skb;
	struct aerial_skb_data		*entry;
	int			retval = 0;
	unsigned long		lockflags;
	size_t			size = dev->rx_urb_size;

	aer_develop("function start.(aerial_rx_submit)\n");

	if ((skb = alloc_skb (size + NET_IP_ALIGN, flags)) == NULL) {
		aer_err ("no rx skb\n");
		aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_RX_MEMORY);
		if (urb) {
			usb_free_urb (urb);
			urb = NULL;
		}
		return;
	}
	skb_reserve (skb, NET_IP_ALIGN);

	entry = (struct aerial_skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->state = aerial_rx_start;
	entry->length = 0;

	usb_fill_bulk_urb (urb, dev->udev, dev->in,
		skb->data, size, aerial_rx_complete, skb);

	spin_lock_irqsave (&dev->rxq.lock, lockflags);

	if (netif_running (dev->net)
			&& netif_device_present (dev->net)
			&& !test_bit (AERIAL_EVENT_RX_HALT, &dev->flags)) {
		switch (retval = usb_submit_urb (urb, GFP_ATOMIC)) {
		case -EPIPE:
			aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_RX_HALT);
			break;
		case -ENOMEM:
			aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_RX_MEMORY);
			break;
		case -ENODEV:
			aer_debug ("device gone\n");
			netif_device_detach (dev->net);
			break;
		default:
			aer_debug ("rx submit, %d\n", retval);
			tasklet_schedule (&dev->bh);
			break;
		case 0:
			__skb_queue_tail (&dev->rxq, skb);
		}
	} else {
		aer_debug ("rx: stopped\n");
		retval = -ENOLINK;
	}
	spin_unlock_irqrestore (&dev->rxq.lock, lockflags);
	if (retval) {
		if (skb) {
			dev_kfree_skb_any (skb);
			skb = NULL;
		}
		if (urb) {
			usb_free_urb (urb);
			urb = NULL;
		}
	}
}


/******************************************************************************
 * aerial_rx_process  -
 *
 *****************************************************************************/
static inline void 
aerial_rx_process (struct aerial_usbnet *dev, struct sk_buff *skb)
{

	aer_develop("function start.(aerial_rx_process )\n");

	if (!aerial_rx_fixup (dev, skb))
		goto error;
	// else network stack removes extra byte if we forced a short packet

	if (skb->len){
		aerial_usbnet_skb_return (dev, skb);
	}
	else {
		aer_err ("drop\n");
error:
		dev->stats.rx_errors++;
		skb_queue_tail (&dev->done, skb);
	}
}

/******************************************************************************
 * aerial_rx_complete  -
 *
 *****************************************************************************/
static void 
aerial_rx_complete (struct urb *urb)
{
	struct sk_buff		*skb = (struct sk_buff *) urb->context;
	struct aerial_skb_data		*entry = (struct aerial_skb_data *) skb->cb;
	struct aerial_usbnet		*dev = entry->dev;
	int			urb_status = urb->status;

	aer_develop("function start.(aerial_rx_complete)\n");

	skb_put (skb, urb->actual_length);
	entry->state = aerial_rx_done;
	entry->urb = NULL;

	if(urb_status)
		aer_develop2 ("aerial_rx_complete error (%d)\n", urb_status );

	switch (urb_status) {
	/* success */
	case 0:
		if (skb->len < dev->net->hard_header_len) {
			aer_err ("rx length error skb=%d, head=%d\n", skb->len, dev->net->hard_header_len );
			entry->state = aerial_rx_cleanup;
			dev->stats.rx_errors++;
			dev->stats.rx_length_errors++;
		}
		break;

	/* stalls need manual reset. this is rare ... except that
	 * when going through USB 2.0 TTs, unplug appears this way.
	 * we avoid the highspeed version of the ETIMEOUT/EILSEQ
	 * storm, recovering as needed.
	 */
	case -EPIPE:
		aer_debug ("aerial_rx_complete urb status error(%d)\n", urb_status );
		dev->stats.rx_errors++;
		aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_RX_HALT);
		// FALLTHROUGH

	/* software-driven interface shutdown */
	case -ECONNRESET:		/* async unlink */
	case -ESHUTDOWN:		/* hardware gone */
	case -EPROTO:
		aer_debug ("aerial_rx_complete urb status error(%d)\n", urb_status );
		goto block;

	/* we get controller i/o faults during khubd disconnect() delays.
	 * throttle down resubmits, to avoid log floods; just temporarily,
	 * so we still recover when the fault isn't a khubd delay.
	 */
	case -ETIME:
	case -EILSEQ:
		aer_debug ("aerial_rx_complete urb status error(%d)\n", urb_status );
		dev->stats.rx_errors++;
		if (!timer_pending (&dev->delay)) {
			mod_timer (&dev->delay, jiffies + THROTTLE_JIFFIES);
			aer_debug ("rx throttle %d\n", urb_status);
		}
block:
		entry->state = aerial_rx_cleanup;
		entry->urb = urb;
		urb = NULL;
		break;

	/* data overrun ... flush fifo? */
	case -EOVERFLOW:
		aer_debug ("aerial_rx_complete urb status error(%d)\n", urb_status );
		dev->stats.rx_over_errors++;
		// FALLTHROUGH

	default:
		aer_debug ("aerial_rx_complete urb status error(%d)\n", urb_status );
		entry->state = aerial_rx_cleanup;
		dev->stats.rx_errors++;
		break;
	}

	aerial_defer_bh(dev, skb, &dev->rxq);

	if (urb) {
		if (netif_running (dev->net)
				&& !test_bit (AERIAL_EVENT_RX_HALT, &dev->flags)) {
			aerial_rx_submit (dev, urb, GFP_ATOMIC);
			return;
		}
		if (urb) {
			usb_free_urb (urb);
			urb = NULL;
		}
	}
}



/******************************************************************************
 * aerial_intr_complete  -
 *
 *****************************************************************************/

static void aerial_intr_complete (struct urb *urb)
{
	struct aerial_private 	*priv = (struct aerial_private*)urb->context;
	struct aerial_usbnet		*dev  = priv->usbnetDev;
	int			status = urb->status;

	aer_develop("function start.(aerial_intr_complete)\n");

	switch (status) {

	/* success */
	case 0:
		;
		;
		; /* Judgment of status */
		;
		;
		break;

	/* software-driven interface shutdown */
	case -ENOENT:		/* urb killed */
	case -ESHUTDOWN:	/* hardware gone */
		aer_debug ("aerial_intr_complete urb status error(%d)\n", status );
		return;

	/* NOTE:  not throttling like RX/TX, since this endpoint
	 * already polls infrequently
	 */
	default:
		aer_err ("aerial_intr_complete urb status error(%d)\n", status );
		break;
	}

	if (!netif_running (dev->net))
		return;

	memset(urb->transfer_buffer, 0, urb->transfer_buffer_length);
	status = usb_submit_urb (urb, GFP_ATOMIC);
	aer_debug ("intr resubmit --> %d", status);
}

/******************************************************************************
 * aerial_unlink_urbs  -
 *
 *****************************************************************************/
static int 
aerial_unlink_urbs (struct aerial_usbnet *dev, struct sk_buff_head *q)
{
	unsigned long		flags;
	struct sk_buff		*skb, *skbnext;
	int			count = 0;

	aer_develop("function start.(aerial_unlink_urbs)\n");

	spin_lock_irqsave (&q->lock, flags);
	for (skb = q->next; skb != (struct sk_buff *) q; skb = skbnext) {
		struct aerial_skb_data		*entry;
		struct urb		*urb;
		int			retval;

		entry = (struct aerial_skb_data *) skb->cb;
		urb = entry->urb;
		skbnext = skb->next;

		// during some PM-driven resume scenarios,
		// these (async) unlinks complete immediately
		retval = usb_unlink_urb (urb);
		if (retval != -EINPROGRESS && retval != 0)
			aer_err ("unlink urb err, %d\n", retval);
		else
			count++;
	}
	spin_unlock_irqrestore (&q->lock, flags);
	return count;
}

/******************************************************************************
 * aerial_kevent  -
 *
 *****************************************************************************/
static void 
aerial_kevent (struct work_struct *work)
{
	struct aerial_usbnet		*dev =
	container_of(work, struct aerial_usbnet, kevent);
	int			status;

	aer_develop("function start.(aerial_kevent)\n");

	/* usb_clear_halt() needs a thread context */
	if (test_bit (AERIAL_EVENT_TX_HALT, &dev->flags)) {
		aerial_unlink_urbs (dev, &dev->txq);
		status = usb_clear_halt (dev->udev, dev->out);
		if (status < 0
				&& status != -EPIPE
				&& status != -ESHUTDOWN) {
			aer_err ( "can't clear tx halt, status %d\n",
					status);
		} else {
			clear_bit (AERIAL_EVENT_TX_HALT, &dev->flags);
			if (status != -ESHUTDOWN)
				netif_wake_queue (dev->net);
		}
	}
	if (test_bit (AERIAL_EVENT_RX_HALT, &dev->flags)) {
		aerial_unlink_urbs (dev, &dev->rxq);
		status = usb_clear_halt (dev->udev, dev->in);
		if (status < 0
				&& status != -EPIPE
				&& status != -ESHUTDOWN) {
			aer_err ( "can't clear rx halt, status %d\n",
					status);
		} else {
			clear_bit (AERIAL_EVENT_RX_HALT, &dev->flags);
			tasklet_schedule (&dev->bh);
		}
	}

	/* tasklet could resubmit itself forever if memory is tight */
	if (test_bit (AERIAL_EVENT_RX_MEMORY, &dev->flags)) {
		struct urb	*urb = NULL;

		if (netif_running (dev->net))
			urb = usb_alloc_urb (0, GFP_KERNEL);
		else
			clear_bit (AERIAL_EVENT_RX_MEMORY, &dev->flags);
		if (urb != NULL) {
			clear_bit (AERIAL_EVENT_RX_MEMORY, &dev->flags);
			aerial_rx_submit (dev, urb, GFP_KERNEL);
			tasklet_schedule (&dev->bh);
		}
	}

	if (dev->flags)
		aer_debug ("aerial_kevent done, flags = 0x%lx\n", dev->flags);
}

/******************************************************************************
 * aerial_tx_complete  -
 *
 *****************************************************************************/
static void 
aerial_tx_complete (struct urb *urb)
{
	struct aerial_data    		*data;
	struct aerial_private 		*priv;
	struct sk_buff				*skb = (struct sk_buff *) urb->context;
	struct aerial_skb_data		*entry = (struct aerial_skb_data *) skb->cb;
	struct aerial_usbnet		*dev = entry->dev;
	data = (struct aerial_data *)dev->data;
	priv =data->priv;

	aer_develop("function start.(aerial_tx_complete)\n");

	if (urb->status == 0) {
		dev->stats.tx_packets++;
		dev->stats.tx_bytes += entry->length;
	} else {
		dev->stats.tx_errors++;

		switch (urb->status) {
		case -EPIPE:
			aer_debug ("aerial_tx_complete urb status error(%d)\n", urb->status );
			aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_TX_HALT);
			break;

		/* software-driven interface shutdown */
		case -ECONNRESET:		// async unlink
		case -ESHUTDOWN:		// hardware gone
			aer_debug ("aerial_tx_complete urb status error(%d)\n", urb->status );
			break;

		// like rx, tx gets controller i/o faults during khubd delays
		// and so it uses the same throttling mechanism.
		case -EPROTO:
		case -ETIME:
		case -EILSEQ:
			aer_debug ("aerial_tx_complete urb status error(%d)\n", urb->status );
			if (!timer_pending (&dev->delay)) {
				mod_timer (&dev->delay,
					jiffies + THROTTLE_JIFFIES);
				aer_err ("tx throttle %d\n",
							urb->status);
			}
			netif_stop_queue (dev->net);
			break;
		default:
			aer_debug ("aerial_tx_complete urb status error(%d)\n", urb->status );
			aer_err ("tx err %d\n", entry->urb->status);
			break;
		}
	}

	urb->dev = NULL;
	entry->state = aerial_tx_done;
	aerial_defer_bh(dev, skb, &dev->txq);
	atomic_dec(&priv->force_active);
}

/******************************************************************************
 * aerial_usbnet_get_endpoints -
 *
 *****************************************************************************/
static int 
aerial_usbnet_get_endpoints(struct aerial_usbnet *dev, struct usb_interface *intf)
{
	int				tmp;
	struct usb_host_interface	*alt = NULL;
	struct usb_host_endpoint	*in = NULL, *out = NULL;
	struct usb_host_endpoint	*status = NULL;
	struct aerial_data    		*data;

	aer_develop("function start.(aerial_usbnet_get_endpoints)\n");

	data = (struct aerial_data *)dev->data;
	for (tmp = 0; tmp < intf->num_altsetting; tmp++) {
		unsigned	ep;
		unsigned	epnum;

		in = out = status = NULL;
		alt = intf->altsetting + tmp;

		aer_debug("aerial_usbnet_get_endpoints tmp=%d, al=%d, epnum=%d\n", tmp, intf->num_altsetting, alt->desc.bNumEndpoints );
		/* take the first altsetting with in-bulk + out-bulk;
		 * remember any status endpoint, just in case;
		 * ignore other endpoints and altsetttings.
		 */
		for (ep = 0; ep < alt->desc.bNumEndpoints; ep++) {
			struct usb_host_endpoint	*e;

			e = alt->endpoint + ep;
			aer_debug("aerial_usbnet_get_endpoints ep=%d, attr=%d\n", ep, e->desc.bmAttributes );

			switch (e->desc.bmAttributes) {
			case USB_ENDPOINT_XFER_INT:
				if (!usb_endpoint_dir_in(&e->desc)){
					aer_debug("Interrupt endpoints error(OUT?)n");
					return -EINVAL;
				}
				tmp_int_in  = e;
				epnum = usb_rcvintpipe (dev->udev,
						e->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
				aer_develop("EP_INTIN ep=%d\n", (epnum>>15)&0x0f );
				break;

			case USB_ENDPOINT_XFER_BULK:
				if (usb_endpoint_dir_in(&e->desc)) {
					epnum = usb_rcvbulkpipe (dev->udev,
							e->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
					switch( (epnum>>15)&0x0f ){
					case AERIAL_EP_DATIN:
						aer_develop("EP_DATIN ep=%d\n", (epnum>>15)&0x0f );
						dev->in = epnum;
						break;
					case AERIAL_EP_WIDIN:
						aer_develop("EP_WIDIN ep=%d\n", (epnum>>15)&0x0f );
						data->wid_bulk_in = epnum;
						break;
					default:
						aer_debug("Bulk-IN endpoints error(EP number=%d)n", (epnum>>15)&0x0f );
						break;
					}
				} else {
					epnum = usb_sndbulkpipe (dev->udev,
							e->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
					switch( (epnum>>15)&0x0f ){
					case AERIAL_EP_DATOUT:
						aer_develop("EP_DATOUT ep=%d\n", (epnum>>15)&0x0f );
						dev->out = epnum;
						break;
					case AERIAL_EP_WIDOUT:
						aer_develop("EP_WIDOUT ep=%d\n", (epnum>>15)&0x0f );
						data->wid_bulk_out = epnum;
						break;
					default:
						aer_debug("Bulk-OUT endpoints error(EP number=%d)n", (epnum>>15)&0x0f );
						break;
					}
				}
				break;
			default:
				continue;
			}
		}
	}

	if (alt->desc.bAlternateSetting != 0
			|| !(dev->driver_info->flags & AERIAL_FLAG_NO_SETINT)) {
		tmp = usb_set_interface (dev->udev, alt->desc.bInterfaceNumber,
				alt->desc.bAlternateSetting);
		if (tmp < 0)
			return tmp;
	}
	return 0;
}



/******************************************************************************
 * aerial_usbnet_init_status  -
 *
 *****************************************************************************/
static int
aerial_usbnet_init_status (struct aerial_usbnet *dev, struct usb_interface *intf)
{
	char			*buf = NULL;
	unsigned		pipe = 0;
	unsigned		maxp;
	unsigned		period;
	struct aerial_data    	*data;
	struct aerial_private 	*priv;
	data = (struct aerial_data *)dev->data;
	priv =data->priv;

	aer_develop("function start.(aerial_usbnet_init_status)\n");

	pipe = usb_rcvintpipe (dev->udev,
			dev->status->desc.bEndpointAddress
				& USB_ENDPOINT_NUMBER_MASK);

	maxp = usb_maxpacket (dev->udev, pipe, 0);

        /* BUG: It changes in v0.3.3 */
	/* avoid 1 msec chatter:  min 8 msec poll rate */
	period = max ((int) dev->status->desc.bInterval,
		(dev->udev->speed == USB_SPEED_HIGH) ? 7 : 3);

	aer_develop("Interrupt IN bInterval=%d\n", period );
	buf = kmalloc (maxp, GFP_KERNEL);
	if (buf) {
		dev->interrupt = usb_alloc_urb (0, GFP_KERNEL);
		if (!dev->interrupt) {
			if (buf) {
				kfree (buf);
				buf = NULL;
			}
			return -ENOMEM;
		} else {
			usb_fill_int_urb(dev->interrupt, dev->udev, pipe,
				buf, maxp, aerial_intr_complete, priv, period);
			aer_debug( "status ep%din, %d bytes period %d\n",
				usb_pipeendpoint(pipe), maxp, period);
		}
	}
	return 0;
}


/******************************************************************************
 * aerial_usbnet_skb_return -
 *
 *****************************************************************************/
static void 
aerial_usbnet_skb_return (struct aerial_usbnet *dev, struct sk_buff *skb)
{
	int	status;

	aer_develop("function start.(aerial_usbnet_skb_return)\n");

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
	skb->dev = dev->net;
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */
	skb->protocol = eth_type_trans (skb, dev->net);
	dev->stats.rx_packets++;
	dev->stats.rx_bytes += skb->len;

	aer_debug ("skb_return: len %zu, type 0x%x\n",
			skb->len + sizeof (struct ethhdr), skb->protocol);
	memset (skb->cb, 0, sizeof (struct aerial_skb_data));
	status = netif_rx (skb);

}



/******************************************************************************
 * aerial_usbnet_get_stats -
 *
 *****************************************************************************/
static struct net_device_stats* 
aerial_usbnet_get_stats (struct net_device *net)
{
	struct aerial_usbnet	*dev = netdev_priv(net);
	aer_develop("function start.(aerial_usbnet_get_stats)\n");
	return &dev->stats;
}

/******************************************************************************
 * aerial_usbnet_defer_kevent -
 *
 *****************************************************************************/
static void 
aerial_usbnet_defer_kevent (struct aerial_usbnet *dev, int work)
{

	aer_develop("function start.(aerial_usbnet_defer_kevent)\n");

	set_bit (work, &dev->flags);
	if (!schedule_work (&dev->kevent))
		aer_err ("kevent %d may have been dropped\n", work);
	else
		aer_develop ("kevent %d scheduled\n", work);
}

/******************************************************************************
 * aerial_usbnet_stop  -
 *
 *****************************************************************************/
static int 
aerial_usbnet_stop (struct net_device *net)
{
	struct aerial_private 	*priv;
	struct aerial_data    	*data;
	struct aerial_usbnet	*dev = netdev_priv(net);
	int						temp;
	DECLARE_WAIT_QUEUE_HEAD_ONSTACK (unlink_wakeup);
	DECLARE_WAITQUEUE (wait, current);

	data = (struct aerial_data *)dev->data;
	priv = data->priv;

	aer_develop("function start.(aerial_usbnet_stop)\n");

	if( atomic_read(&priv->power_mngt)==AERIAL_PWR_MNGT_ACTIVE ){
		netif_stop_queue (net);
	}

	aer_debug ( "stop stats: rx/tx %ld/%ld, errs %ld/%ld\n",
		dev->stats.rx_packets, dev->stats.tx_packets,
		dev->stats.rx_errors, dev->stats.tx_errors
		);

	// ensure there are no more active urbs
	add_wait_queue (&unlink_wakeup, &wait);
	dev->wait = &unlink_wakeup;
	temp = aerial_unlink_urbs (dev, &dev->txq) + aerial_unlink_urbs (dev, &dev->rxq);

	// maybe wait for deletions to finish.
	while (!skb_queue_empty(&dev->rxq)
			&& !skb_queue_empty(&dev->txq)
			&& !skb_queue_empty(&dev->done)) {
		msleep(UNLINK_TIMEOUT_MS);
		aer_debug ("waited for %d urb completions\n", temp);
	}
	dev->wait = NULL;
	remove_wait_queue (&unlink_wakeup, &wait);

	usb_kill_urb(dev->interrupt);

	/* deferred work (task, timer, softirq) must also stop.
	 * can't flush_scheduled_work() until we drop rtnl (later),
	 * else workers could deadlock; so make workers a NOP.
	 */
	dev->flags = 0;
	del_timer_sync (&dev->delay);
	tasklet_kill (&dev->bh);
	usb_autopm_put_interface(dev->intf);
	return 0;
}

/******************************************************************************
 * aerial_usbnet_open  -
 *
 *****************************************************************************/
static int 
aerial_usbnet_open (struct net_device *net)
{
	struct aerial_usbnet		*dev = netdev_priv(net);
	int			retval;
	struct aeria_driver_info	*info = dev->driver_info;

	aer_develop("function start.(aerial_usbnet_open)\n");

	if ((retval = usb_autopm_get_interface(dev->intf)) < 0) {
		aer_info ("resumption fail (%d) usbnet usb-%s-%s, %s\n",
			retval,
			dev->udev->bus->bus_name, dev->udev->devpath,
		info->description);
		goto done_nopm;
	}
	netif_start_queue (net);
	{
		char	*framing = AERIAL_DRVINFO;
		aer_debug ("open: enable queueing "
				"(rx %d, tx %d) mtu %d %s framing\n",
			(int)RX_QLEN (dev), (int)TX_QLEN (dev), dev->net->mtu,
			framing);
	}

	// delay posting reads until we're fully open
	tasklet_schedule (&dev->bh);
	return retval;

done_nopm:
	return retval;
}
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
/******************************************************************************
 * aerial_usb_suspend  -
 *
 *****************************************************************************/
static int
aerial_usbnet_suspend(struct usb_interface *intf, pm_message_t message)
{
	aer_info("%s() called\n", __func__);
	return 0;
}

/******************************************************************************
 * aerial_usb_resume  -
 *
 *****************************************************************************/
static int
aerial_usbnet_resume(struct usb_interface *intf)
{
	aer_info("%s() called\n", __func__);
	return 0;
}
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */
/******************************************************************************
 * aerial_usbnet_tx_timeout  -
 *
 *****************************************************************************/
static void 
aerial_usbnet_tx_timeout (struct net_device *net)
{
	struct aerial_usbnet		*dev = netdev_priv(net);

	aer_develop("function start.(aerial_usbnet_tx_timeout)\n");

	aerial_unlink_urbs (dev, &dev->txq);
	tasklet_schedule (&dev->bh);

	// FIXME: device recovery -- reset?
}

/******************************************************************************
 * aerial_usbnet_start_xmit  -
 *
 *****************************************************************************/
static int 
aerial_usbnet_start_xmit (struct sk_buff *skb, struct net_device *net)
{
	struct aerial_usbnet		*dev = netdev_priv(net);
	struct aerial_data    	*data;
	struct aerial_private 	*priv;
	int			retval = NET_XMIT_SUCCESS;

	data = (struct aerial_data *)dev->data;
	priv = data->priv;

	aer_develop("function start.(aerial_usbnet_start_xmit)\n");
	aerial_wmm_map_and_add_skb(priv, skb);
	
	wake_up_interruptible(&priv->txthr.waitQ);

	return retval;
}

/******************************************************************************
 * aerial_tx_process  -
 *
 *****************************************************************************/
static int
aerial_tx_process(struct sk_buff *skb, struct aerial_usbnet *dev)
{
	int			length;
	int			retval = NET_XMIT_SUCCESS;
	struct urb		*urb = NULL;
	struct aerial_skb_data		*entry;
	unsigned long		flags;

	skb = aerial_tx_fixup (dev, skb, GFP_ATOMIC);
	if (!skb) {
		aer_err ("can't tx_fixup skb\n");
		goto drop;
	}
	length = skb->len;

	if (!(urb = usb_alloc_urb (0, GFP_ATOMIC))) {
		aer_err ("no urb\n");
		goto drop;
	}

	entry = (struct aerial_skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->state = aerial_tx_start;
	entry->length = length;

	usb_fill_bulk_urb (urb, dev->udev, dev->out,
			skb->data, skb->len, aerial_tx_complete, skb);

	spin_lock_irqsave (&dev->txq.lock, flags);

	switch ((retval = usb_submit_urb (urb, GFP_ATOMIC))) {
	case -EPIPE:
		aer_err("The transmission is abnormal.(submit status=%d)\n", retval );
		netif_stop_queue (dev->net);
		aerial_usbnet_defer_kevent (dev, AERIAL_EVENT_TX_HALT);
		break;
	default:
		aer_err("The transmission is abnormal.(submit status=%d)\n", retval );
		break;
	case 0:
		aer_debug("aerial_usbnet_start_xmit success!\n");
		dev->net->trans_start = jiffies;
		__skb_queue_tail (&dev->txq, skb);
		if (dev->txq.qlen >= TX_QLEN (dev))
			netif_stop_queue (dev->net);
	}
	spin_unlock_irqrestore (&dev->txq.lock, flags);

	if (retval) {
		aer_err ("drop, code %d\n", retval);
drop:
		retval = NET_XMIT_SUCCESS;
		dev->stats.tx_dropped++;
		if (skb) {
			dev_kfree_skb_any (skb);
			skb = NULL;
		}
		if (urb) {
			usb_free_urb (urb);
			urb = NULL;
		}
	}
	
	return retval;
}

/******************************************************************************
 * aerial_usbnet_bh  -
 *
 *****************************************************************************/
static void 
aerial_usbnet_bh (unsigned long param)
{
	struct aerial_usbnet		*dev = (struct aerial_usbnet *) param;
	struct sk_buff				*skb;
	struct aerial_skb_data		*entry;
	struct aerial_data			*data;
	struct aerial_private		*priv;

	aer_develop("function start.(aerial_usbnet_bh)\n");

	while ((skb = skb_dequeue (&dev->done))) {
		entry = (struct aerial_skb_data *) skb->cb;
		switch (entry->state) {
		case aerial_rx_done:
			entry->state = aerial_rx_cleanup;
			aerial_rx_process (dev, skb);
			continue;
		case aerial_tx_done:
		case aerial_rx_cleanup:
			aer_debug("aerial_usbnet_bh: %s.\n", (entry->state==aerial_tx_done) ? "tx_done" : "rx_cleanup" );
			if (entry->urb) {
				usb_free_urb (entry->urb);
				entry->urb = NULL;
			}
			if (skb) {
				dev_kfree_skb (skb);
				skb = NULL;
			}
			continue;
		default:
aer_debug ("bogus skb state %d\n", entry->state);
		}
	}


	data = (struct aerial_data *)dev->data;
	priv = data->priv;

	if( atomic_read(&priv->net_suspend_state) ) {
		return;
	}

	// waiting for all pending urbs to complete?
	if (dev->wait) {
		if ((dev->txq.qlen + dev->rxq.qlen + dev->done.qlen) == 0) {
			wake_up (dev->wait);
		}

	// or are we maybe short a few urbs?
	} else if (netif_running (dev->net)
			&& netif_device_present (dev->net)
			&& !timer_pending (&dev->delay)
			&& !test_bit (AERIAL_EVENT_RX_HALT, &dev->flags)) {
		int	temp = dev->rxq.qlen;
		int	qlen = RX_QLEN (dev);

		if (temp < qlen) {
			struct urb	*urb;
			int		i;

			// don't refill the queue all at once
			for (i = 0; i < 10 && dev->rxq.qlen < qlen; i++) {
				urb = usb_alloc_urb (0, GFP_ATOMIC);
				if (urb != NULL) {
					aerial_rx_submit (dev, urb, GFP_ATOMIC);
				}
			}
			if (temp != dev->rxq.qlen )
				aer_debug ("rxqlen %d --> %d\n",
						temp, dev->rxq.qlen);

			if (dev->rxq.qlen < qlen)
				tasklet_schedule (&dev->bh);
		}
		if (dev->txq.qlen < TX_QLEN (dev))
			netif_wake_queue (dev->net);
	}
}



/******************************************************************************
 * aerial_usbnet_disconnect  -
 *
 *****************************************************************************/
static void 
aerial_usbnet_disconnect (struct usb_interface *intf)
{
	struct aerial_usbnet		*dev;
	struct usb_device	*xdev;
	struct net_device	*net;
	struct aerial_data		*data;
	struct aerial_private		*priv;

	aer_develop("function start.(aerial_usbnet_disconnect)\n");

	dev = usb_get_intfdata(intf);
	usb_set_intfdata(intf, NULL);
	if (!dev)
		return;

	xdev = interface_to_usbdev (intf);

	if(!intf->altsetting->desc.bInterfaceNumber){
		tmp_int_in = 0;
		return;
	}

	aer_debug ("unregister '%s' usb-%s-%s, %s\n",
		intf->dev.driver->name,
		xdev->bus->bus_name, xdev->devpath,
		dev->driver_info->description);

	net = dev->net;

	data = (struct aerial_data *)dev->data;
	priv = data->priv;

	aerial_usbnet_clean(dev, intf);
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,29)
static const struct net_device_ops aerial_netdev_ops = {
	.ndo_open =			aerial_usbnet_open,
	.ndo_stop =			aerial_usbnet_stop,
	.ndo_start_xmit =		aerial_usbnet_start_xmit,
	.ndo_tx_timeout =		aerial_usbnet_tx_timeout,
	.ndo_get_stats =		aerial_usbnet_get_stats,
};
#endif

/******************************************************************************
 * aerial_usbnet_probe  -
 *
 *****************************************************************************/
static int
aerial_usbnet_probe (struct usb_interface *intf, const struct usb_device_id *prod)
{
	struct aerial_usbnet		*dev = NULL;
	struct net_device		*net;
	struct usb_host_interface	*interface;
	struct aeria_driver_info	*info;
	struct usb_device		*xdev;
	int				status;
	const char			*name;
	struct aerial_data		*data;
	struct aerial_private		*priv;


#if 0
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,32)
	char mac[AERIAL_MAC_BUF_SIZE];
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,32) */
	DECLARE_MAC_BUF(mac);
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,32) */
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	char mac[AERIAL_MAC_BUF_SIZE];
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
#endif


	aer_develop("function start.(aerial_usbnet_probe)\n");

	name = intf->dev.driver->name;
	info = (struct aeria_driver_info *) prod->driver_info;
	if (!info) {
		dev_dbg (&intf->dev, "blacklisted by %s\n", name);
		return -ENODEV;
	}

	xdev = interface_to_usbdev (intf);
	interface = intf->cur_altsetting;

	usb_get_dev (xdev);

	status = -ENOMEM;

	// set up our own records
	net = alloc_etherdev(sizeof(*dev));
	if (!net) {
		aer_err("can't kmalloc dev(=%u btyes)\n", sizeof(*dev));
		goto out;
	}

	dev = netdev_priv(net);
	dev->udev = xdev;
	dev->intf = intf;
	dev->driver_info = info;
	dev->driver_name = name;
	skb_queue_head_init (&dev->rxq);
	skb_queue_head_init (&dev->txq);
	skb_queue_head_init (&dev->done);
	dev->bh.func = aerial_usbnet_bh;
	dev->bh.data = (unsigned long) dev;
	INIT_WORK (&dev->kevent, aerial_kevent);
	dev->delay.function = aerial_usbnet_bh;
	dev->delay.data = (unsigned long) dev;
	init_timer (&dev->delay);
	mutex_init (&dev->phy_mutex);

	dev->net = net;
	strcpy (net->name, "usb%d");

	/* rx and tx sides can use different message sizes;
	 * bind() should set rx_urb_size in that case.
	 */
	dev->hard_mtu = (AERIAL_BU1805GU_DMA_SIZE*3) + sizeof(struct aerial_packet_header);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,29)
	net->get_stats = aerial_usbnet_get_stats;
	net->hard_start_xmit = aerial_usbnet_start_xmit;
	net->open = aerial_usbnet_open;
	net->stop = aerial_usbnet_stop;
	net->watchdog_timeo = TX_TIMEOUT_JIFFIES;
	net->tx_timeout = aerial_usbnet_tx_timeout;
#else /* LINUX_VERSION_CODE < KERNEL_VERSION(2,6,29) */
	net->netdev_ops = &aerial_netdev_ops;
	net->watchdog_timeo = TX_TIMEOUT_JIFFIES;
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2,6,29) */

	aer_develop("bInterfaceNumber=%d \n", intf->altsetting->desc.bInterfaceNumber );

	if(!intf->altsetting->desc.bInterfaceNumber){
		if((status = aerial_usbnet_get_endpoints(dev,intf)) < 0) {
			goto out1;
		}
		usb_set_intfdata (intf, dev);
		goto out1;
	}
	status = aerial_bind (dev, intf);
	if (status < 0)
		goto out1;

	data = (struct aerial_data *)dev->data;
	priv = data->priv;
	priv->probe_flags.aerial_binded = 1;

	// heuristic:  "usb%d" for links we know are two-host,
	// else "eth%d" when there's reasonable doubt.  userspace
	// can rename the link if it knows better.
	if ((dev->driver_info->flags & AERIAL_FLAG_ETHER) != 0
			&& (net->dev_addr [0] & 0x02) == 0)
		strcpy (net->name, "eth%d");
	/* WLAN devices should always be named "wlan%d" */
	if ((dev->driver_info->flags & AERIAL_FLAG_WLAN) != 0)
		strcpy(net->name, "rohm%d");
	/* maybe the remote can't receive an Ethernet MTU */
	if (net->mtu > (dev->hard_mtu - net->hard_header_len))
		net->mtu = dev->hard_mtu - net->hard_header_len;

	if (dev->status) {
		status = aerial_usbnet_init_status (dev, intf);
		if (status < 0){
			goto out3;
		}
	}

	if (!dev->rx_urb_size)
		dev->rx_urb_size = dev->hard_mtu;
	dev->maxpacket = usb_maxpacket (dev->udev, dev->out, 1);

	SET_NETDEV_DEV(net, &intf->dev);
	status = register_netdev (net);
	if (status)
		goto out3;
	priv->probe_flags.net_registered = 1;
#if 0
	aer_debug ("register '%s' at usb-%s-%s, %s, %s\n",
		intf->dev.driver->name,
		xdev->bus->bus_name, xdev->devpath,
		dev->driver_info->description,

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
		print_mac(mac, net->dev_addr));
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
		aerial_print_mac(mac, net->dev_addr));
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
#endif
	// ok, it's ready to go.
	usb_set_intfdata (intf, dev);

	// start as if the link is up
	netif_device_attach (net);

	status = aerial_last_init (dev);
	if (status != 0)
		goto out4;

	priv->fw_ready_check = 1;
	status = aerial_fw_is_not_ready(priv);
	if (status == 1) {
		/* boot ROM run */
		aer_info("boot rom run\n");
		aer_debug("l.%d:priv->fw_not_ready=%d\n",
			 __LINE__, priv->fw_not_ready);
#ifdef AERIAL_HOTPLUG_FIRMWARE
		aer_info("downloading firmware...\n");
		priv->probe_flags.prev_firmware_found = 0;
		status = aerial_usb_download_fw(intf, priv->firmware_file);
		if (status == (-ENOENT)) {
			aer_err("firmware is nothing\n");
			goto out5;
		}
		if (status == 0) {
			/* if firmware is already downloaded, we should reset
			   that firmware */
			if (priv->probe_flags.prev_firmware_found) {
				aer_info("found previous firmware -- "
					 "resetting...\n");
				/* command hardware reset to device */
				status = aerial_set_f_reset(priv, 3);
				if (status != 0) {
					aer_err("aerial_set_f_reset(3) failed "
						"on %d\n", status);
#if 0
/* free process of this function seems buggy, we prompt user to reset firmware
   manually instead */
aer_err("please execute: iwpriv <netif> set_f_reset 3\n");
#else /* 0/1 */
					goto out5;
#endif /* 0/1 */
				}
			}
			/* device will be disconnected and re-probed soon */
			goto ok;
		}
		else {

			aer_err("aerial_usb_download_fw() failed on %d\n",
				status);
			goto out5;

		}
#endif /* AERIAL_HOTPLUG_FIRMWARE */
	} else if (status == 0) {
		/* firmware run */
		atomic_set(&priv->state, AERIAL_STATE_READY);

		aer_info("new-session firmware run\n");
#ifdef AERIAL_HOTPLUG_FIRMWARE
		/* "iwpriv rohm0 fwsetup" execute here */
		if ((status = aerial_firmware_setup_without_load(priv))
		    != 0) {
			aer_err("aerial_firmware_setup_without_load() "
				"failed on %d\n", status);
			goto out5;
		}
#endif /* AERIAL_HOTPLUG_FIRMWARE */
	} else {
		/* error */
		aer_err("[fatal]:%d: invalid status=%d\n", __LINE__, status);
		goto out5;
	}
	
	aer_info("aerial device successfully probed\n");
	return 0;

out5:
out4:
out3:
out1:
out:
	aerial_usbnet_clean(dev, intf);
	return status;
}

static void
aerial_usbnet_clean (struct aerial_usbnet *dev, struct usb_interface *intf)
{

	struct usb_device		*xdev = NULL;
	struct net_device		*net  = NULL;
	struct aerial_data		*data = NULL;
	struct aerial_private		*priv = NULL;

	if (dev) {
		xdev	= dev->udev;
		net	= dev->net;
		data	= (struct aerial_data *)dev->data;
		priv	= data->priv; 
	}
	
	if (priv) {

		if (priv->fw_ready_check) {
			priv->fw_ready_check = 0;
		}

		if (priv->attr_group.attrs) {
			aerial_sysfs_remove(priv);
		}

		if (priv->probe_flags.net_registered) {
			unregister_netdev (net);
			priv->probe_flags.net_registered = 0;
		}

		if (priv->probe_flags.aerial_binded) {
			aerial_unbind (dev, intf);
		}
	}

	if (net) {
		free_netdev(net);
		net = NULL;
	}

	usb_put_dev(xdev);
}

#ifdef AERIAL_HOTPLUG_FIRMWARE
static int aerial_usb_download_fw(struct usb_interface *intf,
				 const char *filename)
{
	int	err = 0;
	const struct firmware	*fw;
	struct aerial_usbnet	*dev = usb_get_intfdata(intf);
	struct aerial_data	*a_data = (struct aerial_data *)dev->data;
	struct aerial_private	*priv = a_data->priv;

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
	size_t remain_size;
	size_t copy_size;
	int i, loop_max;
	const unsigned char *data;
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

	if (intf == NULL || filename == NULL || filename[0] == '\0')
		return -EINVAL;

	/* check length of firmware_name */
	if (strlen(priv->firmware_file) > (FIRMWARE_NAME_MAX - 1)) {

		aer_err("length of firmware exceeded : \"%s\"=%d "
			"(max length=%d)\n",
			priv->firmware_file, strlen(priv->firmware_file),
			(FIRMWARE_NAME_MAX - 1));

		return -EINVAL;
	}

	err = request_firmware(&fw, filename, &intf->dev);
	if (err) {
		if (err != (-ENOENT)) {
		aer_err("l.%d: request_firmware(\"%s\") failed on %d\n",
			__LINE__, filename, err);
		}
		aer_info("l.%d: request_firmware() failed on %d\n",
			 __LINE__, err);
		goto out;
	}
	
	/* check returned data */
	aer_debug("fw=%p\n", fw);
	if (fw == NULL) {
		err = -EINVAL;
		goto out;
	}
	

	/* copy firmware image to driver */


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	memcpy(priv->fw_image, fw->data, fw->size);
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	remain_size = fw->size;
	data = fw->data;
	/* calculate loop n */
	if (fw->size%FIRMWARE_BLOCK_MAX_SIZE) {
		loop_max = (fw->size/FIRMWARE_BLOCK_MAX_SIZE)+1;
	} else {
		/* futaction */
		loop_max = fw->size/FIRMWARE_BLOCK_MAX_SIZE;
	}

	for (i=0; i<loop_max; i++) {
		if (remain_size/FIRMWARE_BLOCK_MAX_SIZE != 0)
			copy_size = FIRMWARE_BLOCK_MAX_SIZE;
		else
			copy_size = fw->size%FIRMWARE_BLOCK_MAX_SIZE;
		memcpy(priv->fw_image[i], data, copy_size);
		remain_size -= copy_size;
		data += copy_size;
	}
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	/* start download to Aerial module */
	err = aerial_firmware_setup(priv);
	if (err && (priv->fw_ready_check == 0)) {
		aer_err("l.%d: aerial_firmware_setup() failed on %d\n",
			__LINE__, err);
	}
  out:
	release_firmware(fw);
	return err;
}
#endif /* AERIAL_HOTPLUG_FIRMWARE */


#if 0
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
static int aerial_print_mac(char dest[AERIAL_MAC_BUF_SIZE],
			    const unsigned char addr[ETH_ALEN])
{
	int i;
	char *p = dest;
	
	for ( i = 0; i < ETH_ALEN; i++) {
		p += snprintf(p, 3, "%02x", addr[i]);
		if (i >= ETH_ALEN-1) break;
		*p++ = ':';
	}
	return p - dest;
}
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */
#endif

static const struct aeria_driver_info aerial_info = {
	.description = "AERIAL-USB",
	.flags 	= AERIAL_FLAG_WLAN,
};

static const struct usb_device_id	products [] = {
{
	// BU1805  Vendor ID=0x0D4B  ProductID=0x0102
	USB_DEVICE (0x04B5, 0x0102),  
	.driver_info =	(unsigned long) &aerial_info,
}, 
{
	// BU1805(debug)  Vendor ID=0x0D4B  ProductID=0x0112
	USB_DEVICE (0x0D4B, 0x0112),  
	.driver_info =	(unsigned long) &aerial_info,
}, 

	{ },		// END
};
MODULE_DEVICE_TABLE(usb, products);

static struct usb_driver aerial_driver = {
	.name = 	AERIAL_DRVINFO,
	.id_table =	products,
	.probe =	aerial_usbnet_probe,
	.disconnect =	aerial_usbnet_disconnect,

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	.supports_autosuspend = 0,
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	.suspend =	aerial_usbnet_suspend,
	.resume =	aerial_usbnet_resume,
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

};

/******************************************************************************
 * aerial_usbnet_init  -
 *
 *****************************************************************************/
static int __init aerial_usbnet_init(void)
{
	aer_develop("function start.(aerial_usbnet_init)\n");
	return usb_register(&aerial_driver);
}
module_init(aerial_usbnet_init);

/******************************************************************************
 * aerial_usbnet_exit  -
 *
 *****************************************************************************/
static void __exit aerial_usbnet_exit(void)
{
	aer_develop("function start.(aerial_usbnet_exit)\n");
	usb_deregister(&aerial_driver);

}
module_exit(aerial_usbnet_exit);

MODULE_LICENSE("GPL");
