/*
 * drivers/mmc/card/aerial/aerial_sysfs.c
 *
 *  Copyright (C) 2008 Nissin Systems Co.,Ltd.
 *  Copyright (C) 2008-2009 Atmark Techno, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 2008-07-18    Created by Nissin Systems Co.,Ltd.
 * 2009-02-26   Modified by Atmark Techno, Inc.
 */

#include <linux/module.h>

#include "aerial_log.h"
#include "aerial_device.h"
#include "aerial_wid.h"
#include "aerial_fw.h"
#include "aerial_ioctl.h"

struct aerial_attribute {
	struct module_attribute mattr;
	struct aerial_private *priv;

	int flags;
#define AFLAG_NONE	(0)
#define AFLAG_SIGNED	(1)

	int (*get8)(struct aerial_private *, unsigned char *);
	int (*set8)(struct aerial_private *, unsigned char);
	int (*get16)(struct aerial_private *, unsigned short *);
	int (*set16)(struct aerial_private *, unsigned short);
	int (*get32)(struct aerial_private *, unsigned int *);
	int (*set32)(struct aerial_private *, unsigned int);

	int (*getstr)(struct aerial_private *, char *, int);
};

#define to_aerial_attr(x) container_of(x, struct aerial_attribute, mattr)

#define AERIAL_ATTR(_name, _mode, _show, _store)	\
struct aerial_attribute aer_attr_##_name = {		\
	.mattr = __ATTR(_name, _mode, _show, _store),	\
};

#define AERIAL_ATTR_COMMON8(_name, _mode, _get, _set)			\
struct aerial_attribute aer_attr_##_name = {				\
	.mattr = __ATTR(_name, _mode,					\
			aerial_sysfs_common_show,			\
			aerial_sysfs_common_store),			\
	.get8 = (_get),							\
	.set8 = (_set),							\
};
#define AERIAL_ATTR_COMMON8_SIGNED(_name, _mode, _get, _set)		\
struct aerial_attribute aer_attr_##_name = {				\
	.mattr = __ATTR(_name, _mode,					\
			aerial_sysfs_common_show,			\
			aerial_sysfs_common_store),			\
	.get8 = (_get),							\
	.set8 = (_set),							\
	.flags = AFLAG_SIGNED,						\
};
#define AERIAL_ATTR_COMMON16(_name, _mode, _get, _set)			\
struct aerial_attribute aer_attr_##_name = {				\
	.mattr = __ATTR(_name, _mode,					\
			aerial_sysfs_common_show,			\
			aerial_sysfs_common_store),			\
	.get16 = (_get),						\
	.set16 = (_set),						\
};
#define AERIAL_ATTR_COMMON32(_name, _mode, _get, _set)			\
struct aerial_attribute aer_attr_##_name = {				\
	.mattr = __ATTR(_name, _mode,					\
			aerial_sysfs_common_show,			\
			aerial_sysfs_common_store),			\
	.get32 = (_get),						\
	.set32 = (_set),						\
};
#define AERIAL_ATTR_COMMON_STR(_name, _mode, _get)			\
struct aerial_attribute aer_attr_##_name = {				\
	.mattr = __ATTR(_name, _mode,					\
			aerial_sysfs_common_string_show,		\
			NULL),						\
	.getstr = (_get),						\
};

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
static int aerial_strict_strtoul(const char *cp,
			  unsigned int base, unsigned long *res);
static int aerial_strict_strtol(const char *cp, unsigned int base, long *res);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

/******************************************************************************
 * aerial_sysfs_show_firmware -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_show_firmware(struct module_attribute *mattr,
			   struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	int ret;
		ret = aerial_firmware_setup(priv);
		if (ret)
			aer_err("failed firmware load\n");
	return 0;
}

/******************************************************************************
 * aerial_sysfs_store_firmware -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_store_firmware(struct module_attribute *mattr,
			    struct module *mod, const char *buf, size_t count)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	struct aerial_firmware_info *info =
		(struct aerial_firmware_info *)buf;
	static struct aerial_firmware_info header;

	static loff_t pos = 0;

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
	int	block_count;
	size_t	remain_size = 0;
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

	if (info->id == AERIAL_FW_ID) {
		memcpy(&header, buf, sizeof(struct aerial_firmware_info));
		pos = 0;
	}
	
	if (pos + count > FIRMWARE_SIZE_MAX)
		return -EFBIG;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	memcpy(priv->fw_image + pos, buf, count);
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	block_count = pos / FIRMWARE_BLOCK_MAX_SIZE;
	remain_size = FIRMWARE_BLOCK_MAX_SIZE - (pos%FIRMWARE_BLOCK_MAX_SIZE);
	if (remain_size < count) {
		memcpy((priv->fw_image[block_count] +
			(pos%FIRMWARE_BLOCK_MAX_SIZE)),
		       (unsigned char *)buf, remain_size);
		block_count++;
		memcpy(priv->fw_image[block_count], buf+remain_size,
		       count-remain_size);
	}
	else {
		memcpy((priv->fw_image[block_count] +
			(pos % FIRMWARE_BLOCK_MAX_SIZE)),
		       (unsigned char *)buf, count);
	}
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	pos += count;
	priv->fw_size = pos;

	return count;
}

/******************************************************************************
 * aerial_sysfs_show_scanning -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_show_scanning(struct module_attribute *mattr,
			   struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	struct aerial_survey_res survey[2];
	char bar[] = {"---------------------------------------"
		      "---------------------------------------"};
	char fmt[] = {"%-33s|%-5s|%-3d|%-11s|"
		      "%02x.%02x.%02x.%02x.%02x.%02x |%d\n"};
	int len = 0;
	int ret;
	int i, j;


	/* scan AP */
	if ((ret = aerial_start_ap_scan(priv, survey)) != 0)
		return ret;

	len += sprintf(buf + len, "%-33s|%-5s|%-3s|%-11s|%-18s|%-4s\n",
		       "SSID", "Type", "Ch", "Security", "BSSID", "dB");
	len += sprintf(buf + len, "%s\n", bar);

	for (i=0; i<2; i++) {
		int nr_info = (survey[i].size /
			       sizeof(struct aerial_survey_info));

		for (j=0; j<nr_info; j++) {
			struct aerial_survey_info *info;
			char bsstype[][5] = {"BSS", "IBSS"};
			char *security;
			char tmp[16];

			info = &survey[i].info[j];

			/* check data */
			if (info->channel==0)
				continue;

			switch (info->security) {
			case AERIAL_CRYPT_DISABLE:
				security = "NONE"; break;
			case AERIAL_CRYPT_UNKNOWN:
				security = "UNKNOWN"; break;
			case AERIAL_CRYPT_WEP64:
				security = "WEP64"; break;
			case AERIAL_CRYPT_WEP128:
				security = "WEP128"; break;
			case AERIAL_CRYPT_WPA_CCMP:
				security = "WPA-AES"; break;
			case AERIAL_CRYPT_WPA_TKIP:
				security = "WPA-TKIP"; break;
			case AERIAL_CRYPT_WPA_MIX:
				security = "WPA-MIX"; break;
			case AERIAL_CRYPT_WPA2_CCMP:
				security = "WPA2-AES"; break;
			case AERIAL_CRYPT_WPA2_TKIP:
				security = "WPA2-TKIP"; break;
			case AERIAL_CRYPT_WPA2_MIX:
				security = "WPA2-MIX"; break;
			case AERIAL_CRYPT_WPA_2_TKIP:
				security = "WPA/2-TKIP"; break;
			case AERIAL_CRYPT_WPA_2_CCMP:
				security = "WPA/2-AES"; break;
			case AERIAL_CRYPT_WPA_2_MIX:
				security = "WPA/2-MIX"; break;
			default:
				sprintf(tmp, "ID:%02x", info->security);
				security = tmp; break;
			}

			len += sprintf(buf + len, fmt,
				       info->ssid,
				       bsstype[(info->bsstype==0)?0:1],
				       info->channel,
				       security,
				       info->bssid[0],
				       info->bssid[1],
				       info->bssid[2],
				       info->bssid[3],
				       info->bssid[4],
				       info->bssid[5],
				       info->rxpower
				       );
		}
	}

	return len;
}

/******************************************************************************
 * aerial_sysfs_show_versions -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_show_versions(struct module_attribute *mattr,
			   struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	char *print_fmt = "%25s: %s\n";
	char version[AERIAL_VERSION_SIZE + 1];
	int len = 0;
	int ret;
	int i;

	struct info_table {
		int (*fn)(struct aerial_private *, char *, int);
		char *label;
	} table[] = {
		{ aerial_get_firmware_version, "Firmware" },
	};

	len += sprintf(buf + len, print_fmt, "Driver", AERIAL_VERSION);

	for (i=0; i<ARRAY_SIZE(table); i++) {
		memset( version, 0x00, sizeof(version) );
		ret = table[i].fn(priv, version, AERIAL_VERSION_SIZE);
		if (ret)
			return ret;

		len += sprintf(buf + len, print_fmt,
			       table[i].label, version);
	}

	return len;
}

/******************************************************************************
 * aerial_sysfs_show_driver_version -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_show_driver_version(struct module_attribute *mattr,
				 struct module *mod, char *buf)
{
	int len = 0;

	len += sprintf(buf + len, "%s\n", AERIAL_VERSION);

	return len;
}

/******************************************************************************
 * aerial_sysfs_show_wps_cred_list -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_show_wps_cred_list(struct module_attribute *mattr,
			   struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	memcpy(buf, priv->wps_cred_list, priv->wps_cred_list_size);
	aer_devdump( buf, priv->wps_cred_list_size );
	return priv->wps_cred_list_size;
}

/******************************************************************************
 * aerial_sysfs_store_firmware -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_store_wps_cred_list(struct module_attribute *mattr,
			    struct module *mod, const char *buf, size_t count)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	loff_t pos = 0;
	
	if (pos + count > WPS_CRED_LIST_SZ)
		return -EFBIG;

	memcpy(&priv->wps_cred_list[pos], buf, count);
	pos += count;
	priv->wps_cred_list_size = pos;
	aer_devdump( priv->wps_cred_list, priv->wps_cred_list_size );
	return count;
}


/******************************************************************************
 * aerial_sysfs_common_show -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_common_show(struct module_attribute *mattr,
			 struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	int len = 0;
	int ret;

	if (attr->get8) {
		unsigned char val;
		ret = attr->get8(priv, &val);
		if (attr->flags & AFLAG_SIGNED)
			len += sprintf(buf + len, "%d\n", (signed char)val);
		else
			len += sprintf(buf + len, "%d\n", val);
	} else if (attr->get16) {
		unsigned short val;
		ret = attr->get16(priv, &val);
		if (attr->flags & AFLAG_SIGNED)
			len += sprintf(buf + len, "%d\n", (signed short)val);
		else
			len += sprintf(buf + len, "%d\n", val);
	} else if (attr->get32) {
		unsigned int val;
		ret = attr->get32(priv, &val);
		if (attr->flags & AFLAG_SIGNED)
			len += sprintf(buf + len, "%d\n", (signed int)val);
		else
			len += sprintf(buf + len, "%d\n", val);
	} else
		return -EOPNOTSUPP;

	if (ret)
		return ret;

	return len;
}

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
static int aerial_strict_strtoul(const char *cp,
				 unsigned int base, unsigned long *res)
{
	char *tail;
	unsigned long val;
	size_t len;

	*res = 0;
	len = strlen(cp);
	if (len == 0)
		return -EINVAL;

	val = simple_strtoul(cp, &tail, base);
	if (tail == cp)
		return -EINVAL;
	if ((*tail == '\0') ||
		((len == (size_t)(tail - cp) + 1) && (*tail == '\n'))) {
		*res = val;
		return 0;
	}

	return -EINVAL;
}
static int aerial_strict_strtol(const char *cp, unsigned int base, long *res)
{
	int ret;
	if (*cp == '-') {
		ret = aerial_strict_strtoul(cp + 1, base, (unsigned long *)res);
		if (!ret)
			*res = -(*res);
	} else {
		ret = aerial_strict_strtoul(cp, base, (unsigned long *)res);
	}

	return ret;
}
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

/******************************************************************************
 * aerial_sysfs_common_store -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_common_store(struct module_attribute *mattr,
			  struct module *mod, const char *buf, size_t count)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	long val;
	int ret;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	ret = strict_strtol(buf, 0, &val);
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	ret = aerial_strict_strtol(buf, 0, &val);
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */

	if (ret)
		return -EINVAL;

	if (attr->set8) {
		if (val < 0 || 255 < val)
			return -EINVAL;
		ret = attr->set8(priv, (unsigned char)val);
	} else if (attr->set16) {
		if (val < 0 || 65535 < val)
			return -EINVAL;
		ret = attr->set16(priv, (unsigned short)val);
	} else if (attr->set32)
		ret = attr->set32(priv, (unsigned int)val);
	else
		return -EOPNOTSUPP;

	if (ret)
		return ret;

	return count;
}

/******************************************************************************
 * aerial_sysfs_common_string_show -
 *
 *****************************************************************************/
static ssize_t
aerial_sysfs_common_string_show(struct module_attribute *mattr,
				struct module *mod, char *buf)
{
	struct aerial_attribute *attr = to_aerial_attr(mattr);
	struct aerial_private *priv = attr->priv;
	static char val[2048];

	int len = 0;
	int ret;

	if (attr->getstr)
		ret = attr->getstr(priv, val, 2048);
	else
		return -EOPNOTSUPP;

	if (ret)
		return ret;

	len += sprintf(buf + len, "%s\n", val);

	return len;
}

/******************************************************************************
 * aerial_attributes
 *****************************************************************************/
static AERIAL_ATTR(firmware, 0644,
		   aerial_sysfs_show_firmware,
		   aerial_sysfs_store_firmware);
static AERIAL_ATTR(scanning, 0444,
		   aerial_sysfs_show_scanning,
		   NULL);
static AERIAL_ATTR(versions, 0444,
		   aerial_sysfs_show_versions,
		   NULL);
static AERIAL_ATTR_COMMON8(bss_type, 0644,
			   aerial_get_bsstype,
			   aerial_set_bsstype);
static AERIAL_ATTR_COMMON8(current_channel, 0644,
			   aerial_get_channel,
			   aerial_set_channel);
static AERIAL_ATTR_COMMON8(auth_type, 0644,
			   aerial_get_authtype,
			   aerial_set_authtype);
static AERIAL_ATTR_COMMON8(crypt_mode, 0644,
			   aerial_get_cryptmode,
			   aerial_set_cryptmode);
static AERIAL_ATTR_COMMON8(power_management, 0644,
			   aerial_get_powerman,
			   aerial_set_powerman);
static AERIAL_ATTR_COMMON8(scan_type, 0644,
			   aerial_get_scan_type,
			   aerial_set_scan_type);
static AERIAL_ATTR_COMMON8(power_save, 0200,
			   NULL,
			   aerial_set_psctl);
static AERIAL_ATTR_COMMON16(beacon_interval, 0644,
			    aerial_get_beaconint,
			    aerial_set_beaconint);
static AERIAL_ATTR_COMMON8(listen_interval, 0644,
			   aerial_get_lisnintvl,
			   aerial_set_lisnintvl);
static AERIAL_ATTR_COMMON8(stealth, 0644,
			   aerial_get_bcastssid,
			   aerial_set_bcastssid);
static AERIAL_ATTR_COMMON8_SIGNED(rssi, 0444,
				  aerial_get_rssi_u8,
				  NULL);
static AERIAL_ATTR_COMMON8(current_mac_status, 0444,
			   aerial_get_current_mac_status,
			   NULL);
static AERIAL_ATTR_COMMON_STR(firmware_version, 0444,
			      aerial_get_firmware_version);
static AERIAL_ATTR(driver_version, 0444,
		   aerial_sysfs_show_driver_version,
		   NULL);
static AERIAL_ATTR_COMMON8(key_id, 0444,
			   aerial_get_key_id,
			   NULL);
static AERIAL_ATTR_COMMON8(dtim_period, 0644,
			   aerial_get_dtim_period,
			   aerial_set_dtim_period);
static AERIAL_ATTR_COMMON8(rekey_policy, 0644,
			   aerial_get_rekey_policy,
			   aerial_set_rekey_policy);
static AERIAL_ATTR_COMMON32(rekey_period, 0644,
			   aerial_get_rekey_period,
			   aerial_set_rekey_period);
static AERIAL_ATTR_COMMON8(scan_filter, 0644,
			   aerial_get_scan_filter,
			   aerial_set_scan_filter);

#ifndef AERIAL_USB
static AERIAL_ATTR_COMMON32(tx_cis_polling_interval, 0644,
			    aerial_get_txcis_polling_intvl,
			    aerial_set_txcis_polling_intvl);
static AERIAL_ATTR_COMMON32(rx_cis_polling_interval, 0644,
			    aerial_get_rxcis_polling_intvl,
			    aerial_set_rxcis_polling_intvl);
static AERIAL_ATTR_COMMON32(tx_cis_toggle_timeout, 0644,
			    aerial_get_txcis_toggle_timeout,
			    aerial_set_txcis_toggle_timeout);
static AERIAL_ATTR_COMMON32(rx_cis_toggle_timeout, 0644,
			    aerial_get_rxcis_toggle_timeout,
			    aerial_set_rxcis_toggle_timeout);
#else /* AERIAL_USB */
static AERIAL_ATTR_COMMON8(usb_in_xfer_mode, 0644,
			   aerial_get_usb_in_xfer_mode,
			   aerial_set_usb_in_xfer_mode);
#endif /* AERIAL_USB */
static AERIAL_ATTR(wps_cred_list, 0644,
		   aerial_sysfs_show_wps_cred_list,
		   aerial_sysfs_store_wps_cred_list);

static struct aerial_attribute *aerial_attrs[] = {
	&aer_attr_firmware,
	&aer_attr_scanning,
	&aer_attr_versions,
	&aer_attr_bss_type,
	&aer_attr_current_channel,
	&aer_attr_auth_type,
	&aer_attr_crypt_mode,
	&aer_attr_power_management,
	&aer_attr_scan_type,
	&aer_attr_power_save,
	&aer_attr_beacon_interval,
	&aer_attr_listen_interval,
	&aer_attr_stealth,
	&aer_attr_rssi,
	&aer_attr_current_mac_status,
	&aer_attr_firmware_version,
	&aer_attr_driver_version,
	&aer_attr_key_id,
	&aer_attr_dtim_period,
	&aer_attr_rekey_policy,
	&aer_attr_rekey_period,
	&aer_attr_scan_filter,
#ifndef AERIAL_USB
	&aer_attr_tx_cis_polling_interval,
	&aer_attr_rx_cis_polling_interval,
	&aer_attr_tx_cis_toggle_timeout,
	&aer_attr_rx_cis_toggle_timeout,
#else /* AERIAL_USB */
	&aer_attr_usb_in_xfer_mode,
#endif /* AERIAL_USB */
	&aer_attr_wps_cred_list,
};

/******************************************************************************
 * aerial_sysfs_init -
 *
 *****************************************************************************/
int aerial_sysfs_init(struct aerial_private *priv)
{
	int ret = 0;
	int i;

	priv->attr_group.name = priv->netdev->name;
	priv->attr_group.attrs = kzalloc(sizeof(struct attribute *) *
					 (ARRAY_SIZE(aerial_attrs) + 1),
					 GFP_KERNEL);
	if (!priv->attr_group.attrs)
		return -ENOMEM;

	for (i=0; i<ARRAY_SIZE(aerial_attrs); i++) {
		aerial_attrs[i]->priv = priv;
		priv->attr_group.attrs[i] = &aerial_attrs[i]->mattr.attr;
	}

	if (get_device(&priv->netdev->dev)) {
		ret = sysfs_create_group(&THIS_MODULE->mkobj.kobj,
					 &priv->attr_group);
		if (ret) {
			if (priv->attr_group.attrs) {
				kfree(priv->attr_group.attrs);
				priv->attr_group.attrs = NULL;
			}
			return ret;
		}
		put_device(&priv->netdev->dev);
	}

	return 0;
}

/******************************************************************************
 * aerial_sysfs_remove -
 *
 *****************************************************************************/
void aerial_sysfs_remove(struct aerial_private *priv)
{
	sysfs_remove_group(&THIS_MODULE->mkobj.kobj, &priv->attr_group);

	if (priv->attr_group.attrs) {
		kfree(priv->attr_group.attrs);
		priv->attr_group.attrs = NULL;
	}
	return;
}
