/*
 * drivers/mmc/card/aerial/aerial_ioctl.c
 *
 *  Copyright (C) 2008 Nissin Systems Co.,Ltd.
 *  Copyright (C) 2008-2009 Atmark Techno, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 2008-03-05    Created by Nissin Systems Co.,Ltd.
 * 2009-06-19   Modified by Atmark Techno, Inc.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/mutex.h>
#include <linux/workqueue.h>
#ifndef AERIAL_USB
#include <linux/mmc/core.h>
#include <linux/mmc/sdio_func.h>
#include <linux/mmc/sdio_ids.h>
#include <linux/mmc/sdio.h>
#endif /* AERIAL_USB */
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <net/iw_handler.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
#include <linux/ctype.h>
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */


#include "aerial_log.h"
#include "aerial_device.h"
#include "aerial_ioctl.h"
#include "aerial_fw.h"
#include "aerial_wid.h"


#define CHARCODE_TO_NUMBIN(cc, ret)		\
({						\
	unsigned char num;			\
	ret = 0;				\
	switch ((cc)) {				\
	case '0':	/* FALL THROUGH */	\
	case '1':	/* FALL THROUGH */	\
	case '2':	/* FALL THROUGH */	\
	case '3':	/* FALL THROUGH */	\
	case '4':	/* FALL THROUGH */	\
	case '5':	/* FALL THROUGH */	\
	case '6':	/* FALL THROUGH */	\
	case '7':	/* FALL THROUGH */	\
	case '8':	/* FALL THROUGH */	\
	case '9':				\
		num = (cc) - '0'; break;	\
	case 'a':	/* FALL THROUGH */	\
	case 'b':	/* FALL THROUGH */	\
	case 'c':	/* FALL THROUGH */	\
	case 'd':	/* FALL THROUGH */	\
	case 'e':	/* FALL THROUGH */	\
	case 'f':	/* FALL THROUGH */	\
		num = (cc) - 'a' + 10; break;	\
	case 'A':	/* FALL THROUGH */	\
	case 'B':	/* FALL THROUGH */	\
	case 'C':	/* FALL THROUGH */	\
	case 'D':	/* FALL THROUGH */	\
	case 'E':	/* FALL THROUGH */	\
	case 'F':				\
		num = (cc) - 'A' + 10; break;	\
	default:				\
		num = 0; ret = -1; break;	\
	}					\
	num;					\
})

#define NUMBIN_TO_CHARCODE(num)			\
({						\
	char cc;				\
	switch ((num)) {			\
	case 0:		/* FALL THROUGH */	\
	case 1:		/* FALL THROUGH */	\
	case 2:		/* FALL THROUGH */	\
	case 3:		/* FALL THROUGH */	\
	case 4:		/* FALL THROUGH */	\
	case 5:		/* FALL THROUGH */	\
	case 6:		/* FALL THROUGH */	\
	case 7:		/* FALL THROUGH */	\
	case 8:		/* FALL THROUGH */	\
	case 9:					\
		cc = (num) + '0'; break;	\
	case 10:	/* FALL THROUGH */	\
	case 11:	/* FALL THROUGH */	\
	case 12:	/* FALL THROUGH */	\
	case 13:	/* FALL THROUGH */	\
	case 14:	/* FALL THROUGH */	\
	case 15:				\
		cc = (num) - 10 + 'A'; break;	\
	default:				\
		cc = 0;				\
	}					\
	cc;					\
})

static long aerial_support_freq[] = {
	/* 0: no support */
	[ 0] =    0, [ 1] = 2412, [ 2] = 2417, [ 3] = 2422,
	[ 4] = 2427, [ 5] = 2432, [ 6] = 2437, [ 7] = 2442,
	[ 8] = 2447, [ 9] = 2452, [10] = 2457, [11] = 2462,
	[12] = 2467, [13] = 2472,
};

static unsigned char scan_ch_mode = 0x01;
static unsigned char scan_filter  = 0x00;
static unsigned short cur_wait_time= 50;

#ifdef AERIAL_USB
#define netdev_usbpriv(dev)  ({						\
	struct aerial_usbnet *usbnetDev = netdev_priv(dev);			\
	struct aerial_data    *aerialData = (struct aerial_data *)usbnetDev->data; \
	aerialData->priv;})
#define devToPriv  netdev_usbpriv
#else /* AERIAL_USB */
#define devToPriv  netdev_priv
#endif /* AERIAL_USB */

static inline long
_pow(long m, short e)
{
	long r = m;
	int i;
	if (e == 0)
		return 1;
	if (e < 0)
		/* not impremented */
		return -1;
	for (i=1; i<e; i++)
		r *= m;

	return r;
}

static int
support_channel(unsigned char ch)
{
	if ((ch < ARRAY_SIZE(aerial_support_freq))
	    && (aerial_support_freq[ch]))
		return 1; /* supported */

	return 0; /* not supported */
}

static unsigned char
freq_to_channel(struct iw_freq *freq)
{
	int f, power;
	int i;

	/* check parameter */
	if (!freq || (freq->m == 0) ||
	    (freq->e > 1)) {		/* f is more than 10G: error */
		return 0xff; /* not support */
	}

	/* calculate frequency(f)  */
	power = 6 - freq->e;
	f = freq->m / _pow(10, power);

	/* seek */
	for (i=0; i<ARRAY_SIZE(aerial_support_freq); i++)
		if (aerial_support_freq[i] == f)
			return i; /* support */
	return 0xff; /* not support */
}

static int
aerial_iw_not_support(struct net_device *dev, struct iw_request_info *info,
		      void *arg, char *extra)
{
	return -EOPNOTSUPP;
}

static int
aerial_iw_giwname(struct net_device *dev, struct iw_request_info *info,
		  char *name, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	if (atomic_read(&priv->power_state) == AERIAL_PWR_STATE_SLEEP)
		strncpy(name, "sleeping", IFNAMSIZ);
	else
		strncpy(name, "IEEE 802.11bgn", IFNAMSIZ);

	return 0;
}

static int
aerial_iw_siwfreq(struct net_device *dev, struct iw_request_info *info,
		  struct iw_freq *freq, char *extra)
{

	struct aerial_private *priv = devToPriv(dev);

	unsigned char ch;
	int ret;

	aer_debug("l.%d: iw_freq of m=%d,e=%d,i=%u,flags=%u\n", 
		 __LINE__, freq->m, freq->e, freq->i, freq->flags);

	/* convert frequency to channel */
	if ((freq->e == 0) && (freq->m >= 0) && (freq->m <= 1000)) {
		/* f is less than or equal to 1000 : f is treated as channel */
		ch = freq->m;
	}
	else { /* f is more than 1000 : f is treated as freq */
		ch = freq_to_channel(freq);
		
	}

	/* check whether "ch" is supported channel */
	if (!support_channel(ch))
		return -EINVAL;

	/* set channel */
	ret = aerial_set_channel(priv, ch);
	if (ret)
		return ret; /* error */

	return 0; /* success */
}

static int
aerial_iw_giwfreq(struct net_device *dev, struct iw_request_info *info,
		  struct iw_freq *freq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char ch;
	int ret;

	aer_debug("l.%d: iw_freq of m=%d,e=%d,i=%u,flags=%u\n", 
		 __LINE__, freq->m, freq->e, freq->i, freq->flags);

	ret = aerial_get_channel(priv, &ch);
	if (ret)
		return ret;

	if (!support_channel(ch))
		return -EINVAL;

	freq->e = 0;
	freq->m = ch;

	return 0;
}

static int
aerial_iw_iwaplist(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	struct aerial_survey_res survey[2];
	struct sockaddr *addr;
	struct iw_quality quality[IW_MAX_AP], *qual = quality;

	int ret;
	int i,j;
	int n;
	
	/* scan AP */
	if ((ret = aerial_start_ap_scan(priv, survey)) != 0)
		return ret;

	/* return scan data */
	data->length = 0;
	addr = (struct sockaddr*)extra;
	for (i = 0; i < 2; i++) {
		/* number of APs */
		n = survey[i].size/sizeof(struct aerial_survey_info); 

		for (j = 0; j < n; j++) {
			const struct aerial_survey_info	*info =
				&survey[i].info[j];
			
			/* check data */
			if (info->channel==0)
				continue;
			
			/* return BSSID */
			addr->sa_family = ARPHRD_ETHER;
			memcpy(addr->sa_data, info->bssid, sizeof(info->bssid));
			addr++;
			
			/* set link quality */
			/* signeal level (rxpower) */
			if (info->rxpower >= -25)
				qual->level = 100;
			else if (info->rxpower < -25 && info->rxpower >= -30)
				qual->level = 90;
			else if (info->rxpower < -30 && info->rxpower >= -35)
				qual->level = 80;
			else if (info->rxpower < -35 && info->rxpower >= -40)
				qual->level = 70;
			else if (info->rxpower < -40 && info->rxpower >= -45)
				qual->level = 60;
			else if (info->rxpower < -45 && info->rxpower >= -50)
				qual->level = 50;
			else if (info->rxpower < -50 && info->rxpower >= -55)
				qual->level = 45;
			else if (info->rxpower < -55 && info->rxpower >= -60)
				qual->level = 40;
			else if (info->rxpower < -60 && info->rxpower >= -65)
				qual->level = 35;
			else if (info->rxpower < -65 && info->rxpower >= -70)
				qual->level = 30;
			else if (info->rxpower < -70 && info->rxpower >= -75)
				qual->level = 25;
			else if (info->rxpower < -75 && info->rxpower >= -80)
				qual->level = 20;
			else if (info->rxpower < -80 && info->rxpower >= -85)
				qual->level = 15;
			else if (info->rxpower < -85 && info->rxpower >= -90)
				qual->level = 10;
			else if (info->rxpower < -90 && info->rxpower >= -95)
				qual->level = 5;
			else if (info->rxpower < -90 && info->rxpower >= -100)
				qual->level = 1;
			else {
				aer_err("rxpower(=%d) of "
					 "%02x:%02x:%02x:%02x:%02x:%02x  "
					 "is smaller than -100\n",
					 info->rxpower,
					 info->bssid[0], info->bssid[1],
					 info->bssid[2], info->bssid[3],
					 info->bssid[4], info->bssid[5]);
				qual->level = 1;
			}
			qual->noise = 0;	/* noise level */
			qual->qual = 0;		/* quality level */
			qual->updated = 
				IW_QUAL_QUAL_INVALID | IW_QUAL_NOISE_INVALID;
			qual++;
			
			/* increment counter */
			data->length++;
			if (data->length >= IW_MAX_AP) {
				i =INT_MAX-1;
				break;
			}
			
		}
	}
	/* return link quality */
	memcpy(addr, quality, data->length*sizeof(quality[0]));
	data->flags = 1; /* signal quality present (sort of) */
	return 0;

}

static int
aerial_iw_siwmode(struct net_device *dev, struct iw_request_info *info,
		  __u32 *mode, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char type;
	int ret;

	switch (*mode) {
	case IW_MODE_INFRA:
		type = AERIAL_BSS_INFRA; break;
	case IW_MODE_ADHOC:
		type = AERIAL_BSS_ADHOC; break;
	case IW_MODE_AUTO: /* WLAN OFF is treated as auto. */
		type = AERIAL_BSS_WLANOFF; break; 
	case IW_MODE_MASTER:
		type = AERIAL_BSS_AP; break; 
	default:
		return -EOPNOTSUPP;
	}

	ret = aerial_set_bsstype(priv, type);
	if (ret)
		return ret;

	return 0;
}

static int
aerial_iw_giwmode(struct net_device *dev, struct iw_request_info *info,
		  __u32 *mode, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char type;
	int ret;

	ret = aerial_get_bsstype(priv, &type);
	if (ret)
		return ret;

	switch (type) {
	case AERIAL_BSS_INFRA:
		*mode = IW_MODE_INFRA; break;
	case AERIAL_BSS_ADHOC:
		*mode = IW_MODE_ADHOC; break;
	case AERIAL_BSS_WLANOFF: /* WLAN OFF is treated as auto. */
		*mode = IW_MODE_AUTO; break;
	case AERIAL_BSS_AP:
		*mode = IW_MODE_MASTER; break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int
aerial_iw_giwrange(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *data, char *extra)
{
	struct iw_range *range = (struct iw_range *)extra;
	int i;

	memset(range, 0, sizeof(struct iw_range));
	data->length = sizeof(struct iw_range);


	range->num_channels = 0;
	for (i=0; i<ARRAY_SIZE(aerial_support_freq); i++) {
		if (aerial_support_freq[i]) {
			range->freq[range->num_channels].i = i;
			range->freq[range->num_channels].m =
				aerial_support_freq[i];
			range->freq[range->num_channels].e = 6;
			range->num_channels++;
		}
	}
	range->num_frequency = range->num_channels;
	range->num_bitrates = 0;
	range->max_encoding_tokens = AERIAL_KEYINDEX_MAX;
	range->num_encoding_sizes = 2;
	range->encoding_size[0] = 5;  /* 64bits WEP */
	range->encoding_size[1] = 13; /* 128bits WEP */
	range->encoding_size[2] = 32; /* 256bits WPA-PSK */

	range->num_txpower = 0;
	range->txpower_capa = IW_TXPOW_RELATIVE;
	range->min_retry = 0;
	range->max_retry = 0;
	range->max_qual.level = 100;
	range->max_qual.noise = 0;
	range->sensitivity = 0;

	range->min_rts = 256;
	range->max_rts = 2347;
	range->min_frag = 256;
	range->max_frag = 2346;
	range->enc_capa = (IW_ENC_CAPA_WPA |
			   IW_ENC_CAPA_WPA2 |
			   IW_ENC_CAPA_CIPHER_TKIP |
			   IW_ENC_CAPA_CIPHER_CCMP);
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
	range->scan_capa = IW_SCAN_CAPA_ESSID;
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
	range->event_capa[0] = (IW_EVENT_CAPA_K_0 |
				IW_EVENT_CAPA_MASK(SIOCGIWAP) |
				IW_EVENT_CAPA_MASK(SIOCGIWSCAN));
	range->event_capa[1] = IW_EVENT_CAPA_K_1;

	range->we_version_source = SUPPORTED_WIRELESS_EXT;
	range->we_version_compiled = WIRELESS_EXT;

	return 0;
}

static int
aerial_iw_giwap(struct net_device *dev, struct iw_request_info *info,
		struct sockaddr *awrq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char bssid[6];
	int ret;

	ret = aerial_get_bssid(priv, bssid, 6);
	if (ret)
		return ret;

	memcpy(&awrq->sa_data, bssid, 6);
	awrq->sa_family = ARPHRD_ETHER;

	return 0;
}

static int
aerial_iw_siwscan(struct net_device *dev, struct iw_request_info *info,
		  struct iw_point *srq, char *extra)
{
	return 0;
}

static int
aerial_iw_giwscan(struct net_device *dev, struct iw_request_info *info,
		  struct iw_point *srq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	struct aerial_survey_res survey[2];
	struct iw_event iwe;

	char *cev = extra, *cev_end = extra + IW_SCAN_MAX_DATA;
	int ret;
	int i, j;
	char *pret;

	/* scan AP */
	if ((ret = aerial_start_ap_scan(priv, survey)) != 0)
		return ret;

	for (i=0; i<2; i++) {
		int nr_info = (survey[i].size /
			       sizeof(struct aerial_survey_info));
		for (j=0; j<nr_info; j++) {
			struct aerial_survey_info *sv_info;
			sv_info = &survey[i].info[j];

			/* check data */
			if (sv_info->channel==0)
				continue;
			
			/* First entry *MUST* be the BSSID */
			iwe.cmd = SIOCGIWAP;
			iwe.u.ap_addr.sa_family = ARPHRD_ETHER;
			memcpy(iwe.u.ap_addr.sa_data, sv_info->bssid, ETH_ALEN);

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26)
			pret = iwe_stream_add_event(cev, cev_end, &iwe,
						   IW_EV_ADDR_LEN);
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			pret = iwe_stream_add_event(info, cev, cev_end, &iwe,
						   IW_EV_ADDR_LEN);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			/* check returned value */
			if ((cev + IW_EV_ADDR_LEN) != pret) {
				aer_debug("l.%d: "
					  "iwe_stream_add_event() failed\n",
					  __LINE__);
				return -EIO;
			}
			cev = pret;

			/* SSID */
			iwe.cmd = SIOCGIWESSID;
			iwe.u.data.length = strlen(sv_info->ssid);
			iwe.u.data.flags = 1;
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26)
			pret = iwe_stream_add_point(cev, cev_end, &iwe,
						   sv_info->ssid);
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			pret = iwe_stream_add_point(info, cev, cev_end, &iwe,
						   sv_info->ssid);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			/* check returned value */
			if ((cev + IW_EV_POINT_LEN + iwe.u.data.length)
			    != pret) { 
				aer_debug("l.%d: "
					  "iwe_stream_add_point() failed\n",
					  __LINE__);
				return -EIO;
			}
			cev = pret;

			/* Mode */
			iwe.cmd = SIOCGIWMODE;
			switch (sv_info->bsstype) {
			case 0:
				iwe.u.mode = IW_MODE_MASTER;
				break;
			case 1:
				iwe.u.mode = IW_MODE_ADHOC;
				break;
			case 2:
				iwe.u.mode = IW_MODE_AUTO;
				break;
			default:
				aer_err ("invalid bsstype "
					 "WID_SITE_SURVEY_RESULTS: %d\n",
					 sv_info->bsstype);
				return -EPROTONOSUPPORT;
			}
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26)
			pret = iwe_stream_add_event(cev, cev_end, &iwe,
						   IW_EV_UINT_LEN);
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			pret = iwe_stream_add_event(info, cev, cev_end, &iwe,
						   IW_EV_UINT_LEN);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			/* check returned value */
			if ((cev + IW_EV_UINT_LEN) != pret) {
				aer_debug("l.%d: "
					  "iwe_stream_addevent() failed\n",
					  __LINE__);
				return -EIO;
			}
			cev = pret;

			/* Frequency */
			iwe.cmd = SIOCGIWFREQ;
			iwe.u.freq.m = aerial_support_freq[sv_info->channel];
			iwe.u.freq.e = 6;
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26)
			pret = iwe_stream_add_event(cev, cev_end, &iwe,
						   IW_EV_FREQ_LEN);
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			pret = iwe_stream_add_event(info, cev, cev_end, &iwe,
						   IW_EV_FREQ_LEN);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			/* check returned value */
			if ((cev + IW_EV_FREQ_LEN) != pret) {
				aer_debug("l.%d: "
					  "iwe_stream_add_event() failed\n",
					  __LINE__);
				return -EIO;
			}
			cev = pret;

			/* Encryption capability */
			iwe.cmd = SIOCGIWENCODE;
			if (sv_info->security & AERIAL_CRYPT_ENABLE)
				iwe.u.data.flags = (IW_ENCODE_ENABLED |
						    IW_ENCODE_NOKEY);
			else
				iwe.u.data.flags = IW_ENCODE_DISABLED;

			iwe.u.data.length = 0;
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26)
			pret = iwe_stream_add_point(cev, cev_end, &iwe, NULL);
#else /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			pret = iwe_stream_add_point(info, cev, cev_end,
						    &iwe, NULL);
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,26) */
			/* check returned value */
			if ((cev + IW_EV_POINT_LEN + iwe.u.data.length)
			    != pret) {
				aer_debug("l.%d: "
					  "iwe_stream_add_point() failed\n",
					  __LINE__);
				return -EIO;
			}
			cev = pret;
		}
	}

	srq->length = cev - extra;
	srq->flags = 0;

	return 0;
}

static int
aerial_iw_siwessid(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *data, char *essid)
{
	int ret;
	struct aerial_private *priv = devToPriv(dev);

	aer_debug("l.%d: data->flags=0x%x, data->length=%d\n",
		 __LINE__, data->flags, data->length);

	if (data->flags == 0) { /* in case of ANY or OFF */
		data->length = 1; /* set 1 because iwconfig set 0 */
	}

	/* set ESSID string */
	if ((ret = aerial_set_ssid(priv, essid, data->length)) != 0) {
		return ret; /* error */
	}

	return 0;
}

static int
aerial_iw_giwessid(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *data, char *essid)
{
	struct aerial_private *priv = devToPriv(dev);
	int size = min(AERIAL_SSID_MAX, (int)data->length);

	memset(essid, 0, size);
	if (aerial_get_ssid(priv, essid, size) != 0)
		return -EINVAL;

	data->flags = 1;
	data->length = strlen(essid);

	return 0;
}

static int
aerial_iw_siwenc(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *dwrq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char crypt = AERIAL_CRYPT_ENABLE | AERIAL_CRYPT_WEP;
	unsigned char buf[32];
	int index;
	int ret;
	int i;

	aer_debug("l.%d: dwrq->flags=0x%x, dwrq->length=%d\n",
		 __LINE__, dwrq->flags, dwrq->length);

	if ((dwrq->flags & IW_ENCODE_DISABLED) ||
	    (dwrq->flags & IW_ENCODE_OPEN)) {
		return aerial_set_cryptmode(priv, AERIAL_CRYPT_DISABLE);
	}

	if (dwrq->flags & IW_ENCODE_NOKEY)
		return -EINVAL;

	index = dwrq->flags & IW_ENCODE_INDEX;
	if ((index < 0) || (AERIAL_KEYINDEX_MAX < index))
		return -EINVAL;
	if (index != 0)
		index--;

	if (!extra)
		return -EINVAL;
	switch (dwrq->length) {
	case (AERIAL_WEPKEY_MAX / 2):
		crypt |= AERIAL_CRYPT_WEPSZ;
		/* FALL THROUGH */
	case (AERIAL_WEPKEY_MIN / 2):
		memset(buf, 0, sizeof(buf));
		/* convert to ascii code */
		for (i=0; i<dwrq->length; i++) {
			buf[i*2  ] = NUMBIN_TO_CHARCODE((extra[i] >> 4) & 0x0f);
			buf[i*2+1] = NUMBIN_TO_CHARCODE(extra[i] & 0x0f);
		}
		/* preserve */
		memcpy(priv->prev_enckey, buf, sizeof(priv->prev_enckey));
		priv->prev_crypt = crypt;
		break;
	case 0:
		if ((priv->prev_enckey[0] != '\0') && crypt != 0) {
			crypt = priv->prev_crypt;
			memcpy(buf, priv->prev_enckey,
			       sizeof(priv->prev_enckey));
		}
		else {
			aer_err("There is no code key before\n");
			return -EINVAL;
		}
		break;
	default:
		return -EINVAL;
	}

	/* set value to module */
	ret = aerial_set_cryptmode(priv, crypt);
	if (ret)
		return ret;
	ret = aerial_set_wepkey(priv, index, buf, strlen(buf));
	if (ret)
		return ret;

	return 0;
}

static int
aerial_iw_giwenc(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *dwrq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char crypt;
	unsigned char enckey[32];
	int encsize, index;
	int ret, i;

	ret = aerial_get_cryptmode(priv, &crypt);
	if (ret)
		return ret;

	index = dwrq->flags & IW_ENCODE_INDEX;
	if ((index < 0) || (AERIAL_KEYINDEX_MAX < index))
		return -EINVAL;
	if (index != 0)
		index--;

	switch (crypt) {
	case AERIAL_CRYPT_WEP64:
		encsize = AERIAL_WEPKEY_MIN;
		break;
	case AERIAL_CRYPT_WEP128:
		encsize = AERIAL_WEPKEY_MAX;
		break;
	default:
		dwrq->length = 0;
		dwrq->flags = IW_ENCODE_DISABLED;
		return 0;
	}

	memset(enckey, 0, 32);
	ret = aerial_get_wepkey(priv, index, enckey, 32);
	if (ret)
		return ret;

	dwrq->length = encsize / 2;
	dwrq->flags = (index + 1) | IW_ENCODE_ENABLED;
	for (i=0; i<dwrq->length; i++) {
		extra[i] = (char)CHARCODE_TO_NUMBIN(enckey[i*2+1], ret);
		extra[i] |= (char)(CHARCODE_TO_NUMBIN(enckey[i*2], ret) << 4);
	}

	return 0;
}

static int
aerial_iw_siwpower(struct net_device *dev, struct iw_request_info *info,
		   struct iw_param *wrq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char mode;

	if (wrq->disabled) {
		mode = AERIAL_PMMODE_NONE;
	} else {
		switch (wrq->flags & IW_POWER_MODE) {
		case IW_POWER_ON:
			mode = AERIAL_PMMODE_FMIN;
			break;
		default:
			return -EOPNOTSUPP;
		}
	}

	return aerial_set_powerman(priv, mode);
}

static int
aerial_iw_giwpower(struct net_device *dev, struct iw_request_info *info,
		   struct iw_param *rrq, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char val;
	int ret;

	ret = aerial_get_powerman(priv, &val);
	if (ret)
		return ret;

	if (val == AERIAL_PMMODE_NONE)
		rrq->disabled = 1;
	else
		rrq->disabled = 0;

	return 0;
}

static int
aerial_iw_siwauth(struct net_device *dev, struct iw_request_info *info,
		  struct iw_param *dwrq, char *extra)
{
	return 0;
}

static int
aerial_iw_siwencodeext(struct net_device *dev, struct iw_request_info *info,
		       struct iw_point *dwrq, char *extra)
{
	return -EOPNOTSUPP;
}

static int
aerial_iw_giwencodeext(struct net_device *dev, struct iw_request_info *info,
		       struct iw_point *dwrq, char *extra)
{
	return -EOPNOTSUPP;
}

static int
aerial_ip_fwload(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	return aerial_firmware_setup(priv);
}

static int
aerial_ip_fwsetup(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	return aerial_firmware_setup_without_load(priv);
}

static int
aerial_ip_getmac(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char mac[6+1];
	int ret;

	ret = aerial_get_macaddr(priv, mac, 6);
	if (ret)
		return ret;

	sprintf(extra, "%02x%02x%02x%02x%02x%02x",
		mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	data->length = strlen(extra);

	memcpy(priv->netdev->dev_addr, mac, 6);

	return 0;
}

struct aerial_crypt_mode {
	char *label;
	unsigned char val;
};
struct aerial_crypt_mode crypts[] = {
	{ "NONE", 0x00, },
	{ "WEP64", 0x03, },
	{ "WEP128", 0x07, },
	{ "WPA-AES", 0x29 },
	{ "WPA-CCMP", 0x29 },
	{ "WPA-TKIP", 0x49, },
	{ "WPA2-AES", 0x31, },
	{ "WPA2-CCMP", 0x31, },
	{ "WPA2-TKIP", 0x51, },

	{ "WPA-MIX", 0x69, },
	{ "WPA2-MIX", 0x71, },
	{ "WPA/2-AES", 0x39, },
	{ "WPA/2-CCMP", 0x39, },
	{ "WPA/2-TKIP", 0x59, },
	{ "WPA/2-MIX", 0x79, },
};


#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21)
static int aerial_strcasecmp(const char *str1, const char *str2)
{
	int chr1=0, chr2=0;
	
	while (chr1 != 0) {
		chr1 = tolower(*str1++);
		chr2 = tolower(*str2++);
		if (chr1 != chr2)
			break;
	}
	return chr1 - chr2;
}
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,21) */

static int
aerial_ip_setcrypt(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int i;

	for (i=0; i<ARRAY_SIZE(crypts); i++)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21)
		if (strcasecmp(crypts[i].label, extra) == 0)
#else /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
		if (aerial_strcasecmp(crypts[i].label, extra) == 0)
#endif /* LINUX_VERSION_CODE > KERNEL_VERSION(2,6,21) */
			return aerial_set_cryptmode(priv, crypts[i].val);

	return -EINVAL;
}

static int
aerial_ip_getcrypt(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char val;
	int ret, i;

	ret = aerial_get_cryptmode(priv, &val);
	if (ret)
		return ret;

	for (i=0; i<ARRAY_SIZE(crypts); i++) {
		if (crypts[i].val == val) {
			sprintf(extra, "%s\n", crypts[i].label);
			wri->length = strlen(extra);
			return 0;
		}
	}

	sprintf(extra, "unknown(%02x)\n", val);
	wri->length = strlen(extra);
	return 0;
}

static int
aerial_ip_setpsk(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	return aerial_set_passps(priv, extra, wri->length - 1);
}

static int
aerial_ip_getpsk(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	unsigned char pskpass[AERIAL_PSKPASS_MAX + 1];
	int ret;

	ret = aerial_get_passps(priv, pskpass, sizeof(pskpass));
	if (ret)
		return ret;

	strcpy(extra, pskpass);
	wri->length = strlen(extra);

	return 0;
}

#define WID_DEFAULT_TIMEOUT	(1000)
#define WID_GENKEY_TIMEOUT	(20000)
static int
aerial_ip_setwid8(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int 			*param = (int *)extra;
	unsigned short 		wid    = param[0]&0x0000FFFF;
	unsigned char 		val    = param[1]&0x000000FF;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			ret;
	
	if(wid==WID_11I_MODE){
		timeout = WID_GENKEY_TIMEOUT;
	}

	ret = aerial_set_common_value(priv, wid, &val, 1, timeout);
	if (ret){
		aer_err("The setting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The setting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	return 0;
}

static int
aerial_ip_getwid8(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int 			*param = (int *)extra;
	unsigned short 		wid    = param[0]&0x0000FFFF;
	unsigned char 		val    = 0;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			ret;

	ret = aerial_get_common_value(priv, wid, &val, 1, timeout );
	if (ret<0){
		aer_err("The getting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The getting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	param[0] = (int)val;
	return 0;
}


static int
aerial_ip_setwid16(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int 			*param = (int *)extra;
	unsigned short 		wid    = param[0]&0x0000FFFF;
	unsigned short		val    = param[1]&0x0000FFFF;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			ret;

	ret = aerial_set_common_value(priv, wid, &val, 2, timeout);
	if (ret){
		aer_err("The setting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The setting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	return 0;
}

static int
aerial_ip_getwid16(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int 			*param = (int *)extra;
	unsigned short 		wid    = param[0]&0x0000FFFF;
	unsigned short 		val    = 0;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			ret;

	ret = aerial_get_common_value(priv, wid, &val, 2, timeout );
	if (ret<0){
		aer_err("The getting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The getting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	param[0] = (int)val;
	return 0;
}


static int
aerial_ip_setwid32(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char 		*param = (unsigned char *)extra;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	unsigned char*		p1=0;
	unsigned char*		p2=0;
	unsigned short 		wid;
	unsigned int 		val;
	int					ret;
	int					loop,flag;

    param[wri->length]=0x00;
	for( flag=0, loop=0 ; 
	     loop<wri->length && flag<3; 
	     loop++, param++  ){
		switch(flag){
		case 0:
			if(*param!=' ' && *param){
				p1 = param;
				flag++;
			}
			break;
		case 1:
			if(*param==' ' && *param){
				*param = 0x00;
				flag++;
			}
			break;
		case 2:
			if(*param!=' ' && *param){
				p2 = param;
				flag++;
			}
			break;
		}
	}
	if( !p1 || !p2 || flag!=3  ){
		aer_err("Please tie up wid and value by a double quotation. \n" );
		return -EINVAL;
	}

	wid = simple_strtol(p1, (char **)NULL, 16);
	val = simple_strtol(p2, (char **)NULL, 16);

	ret = aerial_set_common_value(priv, wid, &val, 4, timeout);
	if (ret){
		aer_err("The setting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The setting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	return 0;
}

static int
aerial_ip_getwid32(struct net_device *dev, struct iw_request_info *info,
		   struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int 			*param = (int *)extra;
	unsigned short 		wid    = param[0]&0x0000FFFF;
	unsigned int 		val    = 0;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			ret;

	ret = aerial_get_common_value(priv, wid, &val, 4, timeout );
	if (ret<0){
		aer_err("The getting(0x%x) of WID(0x%x) faild. \n", val, wid );
		return -EINVAL;
	}
	aer_info("The getting(0x%x) of WID(0x%x) succeeded. \n", val, wid );
	param[0] = (int)val;
	return 0;
}


static int
aerial_ip_setwidstr(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char 		*param = (unsigned char *)extra;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	unsigned char*		p1=0;
	unsigned char*		p2=0;
	unsigned short 		wid;
	int			size;
	int			ret;
	int			loop,flag;
	static unsigned char	action_req[1024];
	
        param[wri->length]=0x00;
	for( flag=0, loop=0 ; 
	     loop<wri->length && flag<3; 
	     loop++, param++  ){
		switch(flag){
		case 0:
			if(*param!=' ' && *param){
				p1 = param;
				flag++;
			}
			break;
		case 1:
			if(*param==' ' && *param){
				*param = 0x00;
				flag++;
			}
			break;
		case 2:
			if(*param!=' ' && *param){
				p2 = param;
				flag++;
			}
			break;
		}
	}
	if( !p1 || !p2 || flag!=3  ){
		aer_err("Please tie up wid and value by a double quotation. \n" );
		return -EINVAL;
	}

	wid = simple_strtol(p1, (char **)NULL, 16);
	size= strlen(p2);
	
	if(wid==WID_11N_P_ACTION_REQ || wid==WID_11E_P_ACTION_REQ || 
       ((wid & 0xF000)== 0x4000) ){
	    unsigned char sum = 0;
		char  temp[2] = {0,0};
		memset( action_req, 0x00, sizeof(action_req) );
		size = size/2;
		for(loop=0; loop<size; loop++){
			temp[0] = *(p2+(loop*2));
			temp[1] = *(p2+1+(loop*2));
			action_req[loop] = simple_strtol(temp, (char **)NULL, 16);
		}
		
		if( (wid & 0xF000)== 0x4000 ){
		    for( loop=0; loop<size; loop++ ){
		        sum += action_req[loop];
		    }
			action_req[size] = sum;
		}
		p2 = action_req;
		aer_devdump( p2, size );
	}
	else {
		if( wid==WID_SSID || wid==WID_11I_PSK ){
			timeout = WID_GENKEY_TIMEOUT;
		}
	}

	if( (!size) && (size>256) ){
		aer_err("The size is up to 1-256 bytes. (%d)\n", size );
		return -EINVAL;
	}
	ret = aerial_set_common_value(priv, wid, p2, size, timeout);
	if (ret){
		aer_err("The setting of WID(0x%x) faild. \n", wid  );
		return -EINVAL;
	}
	aer_info("The setting of WID(0x%x) succeeded. \n", wid );
	return 0;
}

static int
aerial_ip_getwidstr(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char		*param = (unsigned char *)wri->pointer;
	unsigned int 		timeout= WID_DEFAULT_TIMEOUT;
	int			size;
	unsigned short 		wid;
	static unsigned char 	val[1024];
	int			ret;

	memset(val, 0x00, sizeof(val));
        param[wri->length]=0x00;
	wid = simple_strtol( param, (char **)NULL, 16);
	size= sizeof(val);
	ret = aerial_get_common_value(priv, wid, &val, size, timeout );
	if (ret<0){
		aer_err("The getting of WID(0x%x) faild. \n", wid );
		return -EINVAL;
	}
	size = strlen(val);
	if( (wid == WID_BSSID) || (wid == WID_MAC_ADDR) ) size = 6;
    if( (wid & 0xF000) == 0x4000 ) size = ret;
	aer_info("The getting of WID(0x%x) succeeded. (size=%d)\n", wid, size );
    aerial_dump(LOGLVL_1, &val, size );
//	strcpy(extra, val);
//	wri->length = size;
	wri->length = 0;
	return 0;
}

#define AERIAL_WPS_PIN_LEN 8
#define AERIAL_STRANGE_SIGN		(unsigned char)'?'

static int
aerial_ip_setwpspin(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	char *str = data->pointer;
	unsigned char pin[AERIAL_WPS_PIN_LEN];
	int i;
	
	/* check length */
	if (data->length < AERIAL_WPS_PIN_LEN)
		return -EINVAL;

	for (i = 0; i < AERIAL_WPS_PIN_LEN; i++) {
		/* check range */
		if (str[i] < '0' || str[i] > '9')
			return -EINVAL;
		pin[i] = str[i];
	}

	return  aerial_set_wps_pin(priv, pin, AERIAL_WPS_PIN_LEN);
}

static int
aerial_ip_getwpspin(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	unsigned char pin[8];
	int ret, i;
	
	/* get pincode from device */
	ret = aerial_get_wps_pin(priv, pin, AERIAL_WPS_PIN_LEN);
	if (ret) {
		aer_err("aerial_get_wps_pin() failed on %d\n", ret);
		return ret; /* failure */
	}

	for (i = 0; i < AERIAL_WPS_PIN_LEN; i++) {
		/* check range */
		if (pin[i] <= '9' && pin[i] >= '0')
			extra[i] = pin[i];
		else {
			extra[i] = AERIAL_STRANGE_SIGN;
		}
	}

	/* set length of extra */
	data->length = AERIAL_WPS_PIN_LEN;
	
	return  0; /* success */
}

#define AERIAL_WID_SET_FUNC(_type, _name)				\
do {									\
	{								\
		_type val;						\
									\
		val = param[1];						\
									\
		/* set value to module */				\
		ret = aerial_set_##_name(priv, val);			\
		aer_debug("l.%d: val=%u, ret=%d\n", __LINE__, val, ret);\
		if (ret)						\
			return ret;					\
	}								\
} while(0)


static int
aerial_ioctl_setparam_int(struct net_device *dev, struct iw_request_info *info,
		  void *w, char *extra)
{

	struct aerial_private *priv = devToPriv(dev);
	int *param = (int *)extra;
	int ret;

	switch (param[0])
	{
	case AERIAL_PARAMS_INT_WID_AUTH_TYPE:
		AERIAL_WID_SET_FUNC(unsigned char, authtype);
		break;
	case AERIAL_PARAMS_INT_WID_SITE_SURVEY:
     	aer_info("It doesn't return to Disable(2) after it scans when bit4 is turned on.\n");
        scan_ch_mode = param[1];
		break;
	case AERIAL_PARAMS_INT_WID_JOIN_REQ:
		AERIAL_WID_SET_FUNC(unsigned char, join);
		break;
	case AERIAL_PARAMS_INT_WID_11I_MODE:
		AERIAL_WID_SET_FUNC(unsigned char, cryptmode);
		break;
	case AERIAL_PARAMS_INT_WID_POWER_MANAGEMENT:
		AERIAL_WID_SET_FUNC(unsigned char, powerman);
		break;
	case AERIAL_PARAMS_INT_WID_SCAN_TYPE:
		AERIAL_WID_SET_FUNC(unsigned char, scan_type);
		break;
	case AERIAL_PARAMS_INT_WID_POWER_SAVE:
		AERIAL_WID_SET_FUNC(unsigned char, psctl);
		break;
	case AERIAL_PARAMS_INT_WID_BEACON_INTERVAL:
		AERIAL_WID_SET_FUNC(unsigned short, beaconint);
		break;
	case AERIAL_PARAMS_INT_WID_LISTEN_INTERVAL:
		AERIAL_WID_SET_FUNC(unsigned char, lisnintvl);
		break;
	case AERIAL_PARAMS_INT_WID_BCAST_SSID:
		AERIAL_WID_SET_FUNC(unsigned char, bcastssid);
		break;
	case AERIAL_PARAMS_INT_WID_DTIM_PERIOD:
		AERIAL_WID_SET_FUNC(unsigned char, dtim_period);
		break;
	case AERIAL_PARAMS_INT_WID_REKEY_POLICY:
		AERIAL_WID_SET_FUNC(unsigned char, rekey_policy);
		break;
	case AERIAL_PARAMS_INT_WID_REKEY_PERIOD:
		AERIAL_WID_SET_FUNC(unsigned int, rekey_period);
		break;
	case AERIAL_PARAMS_INT_WID_SCAN_FILTER:
		scan_filter = param[1];
		break;
	case AERIAL_PARAMS_INT_WPSDMD:
		AERIAL_WID_SET_FUNC(unsigned char, wps_dev_mode);
		break;
	case AERIAL_PARAMS_INT_WPSST:
		AERIAL_WID_SET_FUNC(unsigned char, wps_start);
		break;
#ifdef AERIAL_USB
	case AERIAL_PARAMS_INT_WID_USB_IN_XFER_MODE:
		AERIAL_WID_SET_FUNC(unsigned char, usb_in_xfer_mode);
		break;
	case AERIAL_PARAMS_INT_WID_USB_RMTWKUP_TIME:
		AERIAL_WID_SET_FUNC(unsigned char, usb_rmtwkup_time);
		break;
#endif /* AERIAL_USB */
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}


#define AERIAL_WID_GET_FUNC(_type, _name)				\
do {									\
		{							\
		_type val;						\
		/* get value to module */				\
		ret = aerial_get_##_name(priv, &val);			\
		aer_debug("l.%d: val=%u, ret=%d\n", __LINE__, val, ret);\
		if (ret)						\
			return ret;					\
									\
		/* teruen value */					\
		param[0] = val;						\
		return 0;						\
		}							\
} while(0)

static int
aerial_ioctl_getparam_int(struct net_device *dev, struct iw_request_info *info,
			  void *w, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	int *param = (int *)extra; 
	int ret;
	
	switch (param[0])
	{
	case AERIAL_PARAMS_INT_WID_AUTH_TYPE:
		AERIAL_WID_GET_FUNC(unsigned char, authtype);
		break;
	case AERIAL_PARAMS_INT_WID_SITE_SURVEY:
		param[0] = scan_ch_mode;
		break;
	case AERIAL_PARAMS_INT_WID_11I_MODE:
		AERIAL_WID_GET_FUNC(unsigned char, cryptmode);
		break;
	case AERIAL_PARAMS_INT_WID_POWER_MANAGEMENT:
		AERIAL_WID_GET_FUNC(unsigned char, powerman);
		break;
	case AERIAL_PARAMS_INT_WID_SCAN_TYPE:
		AERIAL_WID_GET_FUNC(unsigned char, scan_type);
		break;
	case AERIAL_PARAMS_INT_WID_POWER_SAVE:
        param[0] = (int)atomic_read(&priv->power_state);
		break;
	case AERIAL_PARAMS_INT_WID_BEACON_INTERVAL:
		AERIAL_WID_GET_FUNC(unsigned short, beaconint);
		break;
	case AERIAL_PARAMS_INT_WID_LISTEN_INTERVAL:
		AERIAL_WID_GET_FUNC(unsigned char, lisnintvl);
		break;
	case AERIAL_PARAMS_INT_WID_BCAST_SSID:
		AERIAL_WID_GET_FUNC(unsigned char, bcastssid);
		break;
	case AERIAL_PARAMS_INT_WID_DTIM_PERIOD:
		AERIAL_WID_GET_FUNC(unsigned char, dtim_period);
		break;
	case AERIAL_PARAMS_INT_WID_REKEY_POLICY:
		AERIAL_WID_GET_FUNC(unsigned char, rekey_policy);
		break;
	case AERIAL_PARAMS_INT_WID_REKEY_PERIOD:
		AERIAL_WID_GET_FUNC(unsigned int, rekey_period);
		break;
	case AERIAL_PARAMS_INT_WID_SCAN_FILTER:
		param[0] = scan_filter;
		break;
	case AERIAL_PARAMS_INT_WPSDMD:
		AERIAL_WID_GET_FUNC(unsigned char, wps_dev_mode);
		break;
	case AERIAL_PARAMS_INT_WPSST:
		AERIAL_WID_GET_FUNC(unsigned char, wps_start);
		break;
#ifdef AERIAL_USB
	case AERIAL_PARAMS_INT_WID_USB_IN_XFER_MODE:
		AERIAL_WID_GET_FUNC(unsigned char, usb_in_xfer_mode);
		break;
	case AERIAL_PARAMS_INT_WID_USB_RMTWKUP_TIME:
		AERIAL_WID_GET_FUNC(unsigned char, usb_rmtwkup_time);
		break;
#endif /* AERIAL_USB */
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

static int
aerial_ioctl_setparam_str(struct net_device *dev, struct iw_request_info *info,
    void *w, char *extra)
{
		return -EOPNOTSUPP;
}

#define AERIAL_WID_GET_FUNC_STR(_name)					\
do {									\
	{								\
		memset(str_val, 0x00, sizeof(str_val));			\
									\
		/* get data from module */				\
		ret = aerial_get_##_name(priv,				\
					 str_val, AERIAL_VERSION_SIZE);	\
		if (ret)						\
			return ret;					\
		aer_dump(str_val, sizeof(str_val));			\
									\
		/* return data */					\
		snprintf(extra, sizeof(str_val), "%s" , str_val);	\
		data->length = strlen(extra);				\
	}								\
} while(0)

static int
aerial_ioctl_getparam_str(struct net_device *dev, struct iw_request_info *info,
    void *w, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	struct iw_point *data = (struct iw_point *)w;
	int ret;
	char str_val[AERIAL_VERSION_SIZE + 1];
	

	switch (data->flags)
	{
	case AERIAL_PARAMS_STR_WID_FIRMWARE_VERSION:
		AERIAL_WID_GET_FUNC_STR(firmware_version);
		break;
	case AERIAL_PARAMS_STR_DRIVER_VERSION:
		memset(str_val, 0x00, sizeof(str_val));
		/* return data */
		snprintf(extra, AERIAL_VERSION_SIZE, "%s" , AERIAL_VERSION);
		data->length = strlen(extra);
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}


#ifndef AERIAL_USB
#ifdef AD500_SDIO_GPIO
extern int sdio_gpio_enable;
#endif /* AD500_SDIO_GPIO */

static int
aerial_ip_setsdiogpio(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
#ifdef AD500_SDIO_GPIO
	int *param = (int *)extra;
	int val = param[0];

	sdio_gpio_enable = val;
	return 0;
#else  /* AD500_SDIO_GPIO */
	return -EOPNOTSUPP;
#endif /* AD500_SDIO_GPIO */
}

static int
aerial_ip_getsdiogpio(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
#ifdef AD500_SDIO_GPIO
	int *param = (int *)extra;

	param[0] = sdio_gpio_enable;
	return 0;
#else  /* AD500_SDIO_GPIO */
	return -EOPNOTSUPP;
#endif /* AD500_SDIO_GPIO */

}
#endif  /* AERIAL_USB */

static int
aerial_ip_setlog(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	int *param = (int *)extra;
	int val = param[0];

	aerial_log_flag = val;
	return 0;
}

static int
aerial_ip_getlog(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *wri, char *extra)
{
	int *param = (int *)extra;

	param[0] = aerial_log_flag;

	return 0;
}

static struct iw_statistics *
aerial_iw_getstats(struct net_device *dev)
{
	return NULL;
}


/******************************************************************************
 * aerial_ip_setwps_cred_list -
 *
 *****************************************************************************/
static int
aerial_ip_setwps_cred_list(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);
	
	if(strcmp(data->pointer, "permit")==0){

		if(aerial_set_wps_cred_list(priv) != 0){
			aer_err("failed set WPS Credentials\n");
			return -EINVAL;
		}
		aer_info("Set WPS Credentials success\n");
		return 0;
	}
	aer_info("Please input to make sure as \"permit\".\n");
	return -EINVAL;
}

/******************************************************************************
 * aerial_ip_getwps_cred_list -
 *
 *****************************************************************************/

static int
aerial_ip_getwps_cred_list(struct net_device *dev, struct iw_request_info *info,
		 struct iw_point *data, char *extra)
{
	struct aerial_private *priv = devToPriv(dev);

	data->length = 0;
	if(strcmp(data->pointer, "permit")==0){

		if(aerial_get_wps_cred_list(priv) < 0){
			aer_err("failed get WPS Credentials\n");
			return -EINVAL;
		}
		aer_info("Get WPS Credentials success\n");
		return 0;
	}
	aer_info("Please input to make sure as \"permit\".\n");
	return -EINVAL;
}

#define IW_IOCTL(x, func) [(x) - SIOCSIWCOMMIT] = (iw_handler)func
static const iw_handler aerial_iw_handlers[] = {
	/* Wireless Identification */
	IW_IOCTL(SIOCSIWCOMMIT,		aerial_iw_not_support),
	IW_IOCTL(SIOCGIWNAME,		aerial_iw_giwname),
	/* Basic operations */
	IW_IOCTL(SIOCSIWNWID,		NULL),
	IW_IOCTL(SIOCGIWNWID,		aerial_iw_not_support),
	IW_IOCTL(SIOCSIWFREQ,		aerial_iw_siwfreq),
	IW_IOCTL(SIOCGIWFREQ,		aerial_iw_giwfreq),
	IW_IOCTL(SIOCSIWMODE,		aerial_iw_siwmode),
	IW_IOCTL(SIOCGIWMODE,		aerial_iw_giwmode),
	IW_IOCTL(SIOCSIWSENS,		NULL),
	IW_IOCTL(SIOCGIWSENS,		aerial_iw_not_support),
	/* Informative stuff */
	IW_IOCTL(SIOCSIWRANGE,		NULL),
	IW_IOCTL(SIOCGIWRANGE,		aerial_iw_giwrange),
	IW_IOCTL(SIOCSIWPRIV,		NULL),
	IW_IOCTL(SIOCGIWPRIV,		NULL),
	IW_IOCTL(SIOCSIWSTATS,		NULL),
	IW_IOCTL(SIOCGIWSTATS,		NULL),
	/* Spy support */
	IW_IOCTL(SIOCSIWSPY,		NULL),
	IW_IOCTL(SIOCGIWSPY,		NULL),
	IW_IOCTL(SIOCSIWTHRSPY,		NULL),
	IW_IOCTL(SIOCGIWTHRSPY,		NULL),
	/* Access Point manipulation */
	IW_IOCTL(SIOCSIWAP,			NULL),
	IW_IOCTL(SIOCGIWAP,			aerial_iw_giwap),
	IW_IOCTL(SIOCGIWAPLIST,		aerial_iw_iwaplist),
	IW_IOCTL(SIOCSIWSCAN,		aerial_iw_siwscan),
	IW_IOCTL(SIOCGIWSCAN,		aerial_iw_giwscan),
	/* 802.11 specific support */
	IW_IOCTL(SIOCSIWESSID,		aerial_iw_siwessid),
	IW_IOCTL(SIOCGIWESSID,		aerial_iw_giwessid),
	IW_IOCTL(SIOCSIWNICKN,		NULL),
	IW_IOCTL(SIOCGIWNICKN,		NULL),
	/* Other parameters useful in 802.11 and some other devices */
	IW_IOCTL(SIOCSIWRATE,		NULL),
	IW_IOCTL(SIOCGIWRATE,		aerial_iw_not_support),
	IW_IOCTL(SIOCSIWRTS,		NULL),
	IW_IOCTL(SIOCGIWRTS,		aerial_iw_not_support),
	IW_IOCTL(SIOCSIWFRAG,		NULL),
	IW_IOCTL(SIOCGIWFRAG,		aerial_iw_not_support),
	IW_IOCTL(SIOCSIWTXPOW,		NULL),
	IW_IOCTL(SIOCGIWTXPOW,		aerial_iw_not_support),
	IW_IOCTL(SIOCSIWRETRY,		NULL),
	IW_IOCTL(SIOCGIWRETRY,		aerial_iw_not_support),
	/* Encoding stuff (scrambling, hardware security, WEP...) */
	IW_IOCTL(SIOCSIWENCODE,		aerial_iw_siwenc),
	IW_IOCTL(SIOCGIWENCODE,		aerial_iw_giwenc),
	/* Power saving stuff (power management, unicast and multicast) */
	IW_IOCTL(SIOCSIWPOWER,		aerial_iw_siwpower),
	IW_IOCTL(SIOCGIWPOWER,		aerial_iw_giwpower),
	/* WPA : Generic IEEE 802.11 informatiom element */
	IW_IOCTL(SIOCSIWGENIE,		NULL),
	IW_IOCTL(SIOCGIWGENIE,		NULL),
	/* WPA : IEEE 802.11 MLME requests */
	IW_IOCTL(SIOCSIWMLME,		NULL),
	/* WPA : Authentication mode parameters */
	IW_IOCTL(SIOCSIWAUTH,		aerial_iw_siwauth),
	IW_IOCTL(SIOCGIWAUTH,		NULL),
	/* WPA : Extended version of encoding configuration */
	IW_IOCTL(SIOCSIWENCODEEXT,	aerial_iw_siwencodeext),
	IW_IOCTL(SIOCGIWENCODEEXT,	aerial_iw_giwencodeext),
	/* WPA2 : PMKSA cache management */
	IW_IOCTL(SIOCSIWPMKSA,		NULL),
};

#define IW_PRIV(x, func) [(x) - SIOCIWFIRSTPRIV] = (iw_handler)func
static const iw_handler aerial_iw_priv_handlers[] = {
	IW_PRIV(AERIAL_PRIV_FWLOAD,		aerial_ip_fwload),
	IW_PRIV(AERIAL_PRIV_FWSETUP,	aerial_ip_fwsetup),
	IW_PRIV(AERIAL_PRIV_GETMAC,		aerial_ip_getmac),
	IW_PRIV(AERIAL_PRIV_SETCRYPT,	aerial_ip_setcrypt),
	IW_PRIV(AERIAL_PRIV_GETCRYPT,	aerial_ip_getcrypt),
	IW_PRIV(AERIAL_PRIV_SETPSK,	aerial_ip_setpsk),
	IW_PRIV(AERIAL_PRIV_GETPSK,	aerial_ip_getpsk),

	IW_PRIV(AERIAL_PRIV_SETWID8,	aerial_ip_setwid8),
	IW_PRIV(AERIAL_PRIV_GETWID8,	aerial_ip_getwid8),
	IW_PRIV(AERIAL_PRIV_SETWID16,	aerial_ip_setwid16),
	IW_PRIV(AERIAL_PRIV_GETWID16,	aerial_ip_getwid16),
	IW_PRIV(AERIAL_PRIV_SETWID32,	aerial_ip_setwid32),
	IW_PRIV(AERIAL_PRIV_GETWID32,	aerial_ip_getwid32),
	IW_PRIV(AERIAL_PRIV_SETWIDSTR,	aerial_ip_setwidstr),
	IW_PRIV(AERIAL_PRIV_GETWIDSTR,	aerial_ip_getwidstr),

	IW_PRIV(AERIAL_PRIV_SETWPSPIN,	aerial_ip_setwpspin),
	IW_PRIV(AERIAL_PRIV_GETWPSPIN,	aerial_ip_getwpspin),
	IW_PRIV(AERIAL_IOCTL_SETPARAM_INT,	aerial_ioctl_setparam_int),
	IW_PRIV(AERIAL_IOCTL_GETPARAM_INT,	aerial_ioctl_getparam_int),
	IW_PRIV(AERIAL_IOCTL_SETPARAM_STR,	aerial_ioctl_setparam_str),
	IW_PRIV(AERIAL_IOCTL_GETPARAM_STR,	aerial_ioctl_getparam_str),
	IW_PRIV(AERIAL_PRIV_SETWPSCREDLIST,	aerial_ip_setwps_cred_list),
	IW_PRIV(AERIAL_PRIV_GETWPSCREDLIST,	aerial_ip_getwps_cred_list),
#ifndef AERIAL_USB
	IW_PRIV(AERIAL_PRIV_SETSDIO_GPIO,	aerial_ip_setsdiogpio),
	IW_PRIV(AERIAL_PRIV_GETSDIO_GPIO,	aerial_ip_getsdiogpio),
#endif  /* AERIAL_USB */
	IW_PRIV(AERIAL_PRIV_SETLOG,	aerial_ip_setlog),
	IW_PRIV(AERIAL_PRIV_GETLOG,	aerial_ip_getlog),
};

#define IW_PRIV_ARGS(_cmd, _name, _set, _get) \
{                                             \
	.cmd = _cmd,                          \
	.name = _name,                        \
	.set_args = _set,                     \
	.get_args = _get,                     \
}
#define IW_ARG_CHAR_T(x) (IW_PRIV_TYPE_CHAR | (x))
#define IW_ARG_INT_FIXED_T(x) (IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | (x))
#define IW_ARG_CHAR_FIXED_T(x) (IW_PRIV_TYPE_CHAR  | IW_PRIV_SIZE_FIXED | (x))
#define IW_ARG_BYTE_T(x) (IW_PRIV_TYPE_BYTE | (x))
static const struct iw_priv_args aerial_iw_priv_args[] = {
	IW_PRIV_ARGS(AERIAL_PRIV_FWLOAD, "fwload",
		     0,
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_FWSETUP, "fwsetup",
		     0,
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETMAC, "get_macaddr",
		     0,
		     IW_ARG_CHAR_T(12)),
	IW_PRIV_ARGS(AERIAL_PRIV_SETCRYPT, "set_cryptmode",
		     IW_ARG_CHAR_T(32),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETCRYPT, "get_cryptmode",
		     0,
		     IW_ARG_CHAR_T(32)),
	IW_PRIV_ARGS(AERIAL_PRIV_SETPSK, "set_psk",
		     IW_ARG_CHAR_T(128),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETPSK, "get_psk",
		     0,
		     IW_ARG_CHAR_T(128)),

	IW_PRIV_ARGS(AERIAL_PRIV_SETWID8, "set_wid_8",
		     IW_ARG_INT_FIXED_T(2),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWID8, "get_wid_8",
		     IW_ARG_INT_FIXED_T(1),
		     IW_ARG_INT_FIXED_T(1)),

	IW_PRIV_ARGS(AERIAL_PRIV_SETWID16, "set_wid_16",
		     IW_ARG_INT_FIXED_T(2),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWID16, "get_wid_16",
		     IW_ARG_INT_FIXED_T(1),
		     IW_ARG_INT_FIXED_T(1)),

	IW_PRIV_ARGS(AERIAL_PRIV_SETWID32, "set_wid_32",
		     IW_ARG_CHAR_T(32),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWID32, "get_wid_32",
		     IW_ARG_INT_FIXED_T(1),
		     IW_ARG_INT_FIXED_T(1)),

	IW_PRIV_ARGS(AERIAL_PRIV_SETWIDSTR, "set_wid_str",
		     IW_ARG_CHAR_T(2000+16),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWIDSTR, "get_wid_str",
		     IW_ARG_CHAR_T(16),
		     IW_ARG_CHAR_T(2000)),

	IW_PRIV_ARGS(AERIAL_PRIV_SETWPSPIN, "set_wpspin",
		     IW_ARG_CHAR_T(8),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWPSPIN, "get_wpspin",
		     0,
		     IW_ARG_CHAR_T(8)),

    /*
     * These depends on sub-ioctl support which added in version 12.
     */
    /* sub-ioctl definitions */
	IW_PRIV_ARGS(AERIAL_IOCTL_SETPARAM_INT, "",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_IOCTL_GETPARAM_INT, "",
		     0,
		     IW_ARG_INT_FIXED_T(1)),

	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_AUTH_TYPE,
		     "set_auth_type",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_AUTH_TYPE,
		     "auth_type",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SITE_SURVEY,
		     "set_scan_chmode",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SITE_SURVEY,
		     "scan_chmode",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_JOIN_REQ,
		     "set_join_req",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_JOIN_REQ,
		     "",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_11I_MODE,
		     "set_11i_mode",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_11I_MODE,
		     "11i_mode",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_POWER_MANAGEMENT,
		     "set_power_man",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_POWER_MANAGEMENT,
		     "power_man",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SCAN_TYPE,
		     "set_scan_type",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SCAN_TYPE,
		     "scan_type",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_POWER_SAVE,
		     "set_pow_save",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_POWER_SAVE,
		     "pow_save",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_BEACON_INTERVAL,
		     "set_beacon_int",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_BEACON_INTERVAL,
		     "beacon_int",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_LISTEN_INTERVAL,
		     "set_listen_int",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_LISTEN_INTERVAL,
		     "listen_int",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_BCAST_SSID,
		     "set_bcast_ssid",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_BCAST_SSID,
		     "bcast_ssid",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_DTIM_PERIOD,
		     "set_dtim_period",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_DTIM_PERIOD,
		     "dtim_period",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_REKEY_POLICY,
		     "set_rkey_policy",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_REKEY_POLICY,
		     "rkey_policy",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_REKEY_PERIOD,
		     "set_rkey_period",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_REKEY_PERIOD,
		     "rkey_period",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SCAN_FILTER,
		     "set_scan_filter",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_SCAN_FILTER,
		     "scan_filter",
		     0,
		     IW_ARG_INT_FIXED_T(1)),

	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WPSDMD,
		     "set_wpsdmd",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WPSDMD,
		     "get_wpsdmd",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WPSST,
		     "set_wpsst",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WPSST,
		     "get_wpsst",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
#ifdef AERIAL_USB
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_USB_RMTWKUP_TIME,
		     "set_rmt_wkup",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_USB_RMTWKUP_TIME,
		     "rmt_wkup",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_USB_IN_XFER_MODE,
		     "set_usb_in_mode",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_INT_WID_USB_IN_XFER_MODE,
		     "usb_in_mode",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
#endif /* AERIAL_USB */

    /*
     * These depends on sub-ioctl support which added in version 12.
     */
    /* sub-ioctl handlers (WID command)*/
	IW_PRIV_ARGS(AERIAL_IOCTL_SETPARAM_STR, "",
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE),
		     0),
	IW_PRIV_ARGS(AERIAL_IOCTL_GETPARAM_STR, "",
		     0,
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE)),

	IW_PRIV_ARGS(AERIAL_PARAMS_STR_WID_FIRMWARE_VERSION,
		     "",
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_STR_WID_FIRMWARE_VERSION,
		     "fw_ver",
		     0,
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE)),
	IW_PRIV_ARGS(AERIAL_PARAMS_STR_DRIVER_VERSION,
		     "",
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE),
		     0),
	IW_PRIV_ARGS(AERIAL_PARAMS_STR_DRIVER_VERSION,
		     "driver_ver",
		     0,
		     IW_ARG_CHAR_T(AERIAL_VERSION_SIZE)),


	IW_PRIV_ARGS(AERIAL_PRIV_SETWPSCREDLIST, "set_wpslist",
		     IW_ARG_CHAR_T(8),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETWPSCREDLIST, "get_wpslist",
		     IW_ARG_CHAR_T(8),
		     IW_ARG_CHAR_T(8)),

#ifndef AERIAL_USB
	IW_PRIV_ARGS(AERIAL_PRIV_SETSDIO_GPIO,
		     "set_sdio_gpio",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETSDIO_GPIO,
		     "get_sdio_gpio",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
#endif  /* AERIAL_USB */

	IW_PRIV_ARGS(AERIAL_PRIV_SETLOG,
		     "set_log_level",
		     IW_ARG_INT_FIXED_T(1),
		     0),
	IW_PRIV_ARGS(AERIAL_PRIV_GETLOG,
		     "get_log_level",
		     0,
		     IW_ARG_INT_FIXED_T(1)),
};

static struct iw_handler_def aerial_iw_handler_def = {
	.standard = (iw_handler *)aerial_iw_handlers,
	.num_standard = ARRAY_SIZE(aerial_iw_handlers),
	.private = (iw_handler *)aerial_iw_priv_handlers,
	.num_private = ARRAY_SIZE(aerial_iw_priv_handlers),
	.private_args = (struct iw_priv_args *)aerial_iw_priv_args,
	.num_private_args = ARRAY_SIZE(aerial_iw_priv_args),
	.get_wireless_stats = aerial_iw_getstats,
};

void
aerial_wireless_attach(struct net_device *dev)
{
	dev->wireless_handlers = &aerial_iw_handler_def;

	return;
}


extern int
aerial_start_ap_scan(struct aerial_private *priv,
		     struct aerial_survey_res *survey)
{
	unsigned char old_scantype, old_scanfilter;
	int ret;
	int i;
	
	/* backup current parameters for scan */
	if ((ret = aerial_get_scan_type(priv, &old_scantype)) != 0)
		return ret;
	if ((ret = aerial_get_scan_filter(priv, &old_scanfilter)) != 0)
		return ret;

	/* set parameters for scan */
	if ((ret = aerial_set_scan_type(priv, 0x0)) != 0)
		return ret;
	if ((ret = aerial_set_scan_filter(priv, scan_filter)) != 0)
		goto restore_san_type;

	/* start scan for all channels */
	if ((ret = aerial_set_sitesv(priv, scan_ch_mode&0x3 )) != 0)
		goto restore_parameters;
	if ((aerial_set_startscan(priv, 0x01)) != 0)
		goto restore_parameters;

	/* wait for firmware to finish scan */
	for (i=0; i < ARRAY_SIZE(aerial_support_freq); i++) {

		msleep(cur_wait_time);
	}

	/* get result of scan for first half and second half */
	for (i=0; i<2; i++) {
		int size = sizeof(struct aerial_survey_res);
		memset(&survey[i], 0, size);
		ret = aerial_get_site_survey_results(priv, &survey[i], size);
		if (ret){
			ret = -EIO;
			goto restore_parameters;
		}
	}

	if( !(scan_ch_mode & 0x10) ) {
		if ((ret = aerial_set_sitesv(priv, 0x02)) != 0)
			goto restore_parameters;
	}

	/* restore parameters */
	if ((ret = aerial_set_scan_filter(priv, old_scanfilter)) != 0)
		goto restore_parameters;
	if ((ret = aerial_set_scan_type(priv, old_scantype)) != 0)
		goto restore_san_type;
	return 0;
	
restore_parameters:
	aerial_set_scan_filter(priv, old_scanfilter);
restore_san_type:
	aerial_set_scan_type(priv, old_scantype);
	
	return ret;
}

