/*
 * drivers/mmc/card/aerial/aerial_sdiodrv.c
 *
 *  Copyright (C) 2008 Nissin Systems Co.,Ltd.
 *  Copyright (C) 2008-2009 Atmark Techno, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 2008-03-05    Created by Nissin Systems Co.,Ltd.
 * 2009-06-19   Modified by Atmark Techno, Inc.
 */

#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/etherdevice.h>
#include <linux/leds.h>
#include <linux/mmc/host.h>
#include <linux/mmc/card.h>
#include <linux/mmc/sdio.h>
#include <linux/mmc/sdio_func.h>
#include <linux/mmc/sdio_ids.h>

#include "aerial_device.h"
#include "aerial_log.h"
#include "aerial_wid.h"
#include "aerial_ioctl.h"
#include "aerial_fw.h"
#include "aerial_sysfs.h"

#ifdef AD500_SDIO_GPIO
#include <asm/arch/gpio.h>
#define  GPIO_TX_ENABLE		0x01
#define  GPIO_RX_ENABLE		0x02
int      sdio_gpio_enable = 0;
#define  AERIAL_IO_TX_CISSTAT    MX31_PIN_GPIO3_0		/* CON16 : PIN13 (GPIO0) */
#define  AERIAL_IO_TX_TOGGLE     MX31_PIN_GPIO3_1		/* CON16 : PIN14 (GPIO1) */
#define  AERIAL_IO_RX_CISSTAT    MX31_PIN_ATA_CS0		/* CON16 : PIN17 (GPIO6) */
#define  AERIAL_IO_RX_TOGGLE     MX31_PIN_ATA_DIOR		/* CON16 : PIN19 (GPIO2) */

//#define  DISABLE_GPIO_RX_STATUS

#define  IS_IO_TXWAIT()		(mxc_get_gpio_datain(AERIAL_IO_TX_CISSTAT))
#define  IS_IO_TXTOGGLE(t)	((t==TOGGLE_NONE) ? 1 : ((mxc_get_gpio_datain(AERIAL_IO_TX_TOGGLE))==t) )
#define  SET_IO_TOGGLE(t)	{ t ^= 0x01; }

#define  IS_IO_RXRDY()		(mxc_get_gpio_datain(AERIAL_IO_RX_CISSTAT))
#define  IS_IO_RXTOGGLE(t)	((t==TOGGLE_NONE) ? 1 : ((mxc_get_gpio_datain(AERIAL_IO_RX_TOGGLE))==t) )
#endif /* AD500_SDIO_GPIO */

#define  IS_TXWAIT(v)		((v&AERIAL_CISSTAT_MASK)==AERIAL_CISSTAT_TXWAIT)
#define  IS_TXBUSY(v)		(((v&AERIAL_CISSTAT_MASK)==AERIAL_CISSTAT_TXSLEEP) || \
				 ((v&AERIAL_CISSTAT_MASK)==AERIAL_CISSTAT_TXACTIVE) )
#define  IS_RXRDY(v)		((v&AERIAL_CISSTAT_MASK)==AERIAL_CISSTAT_RXREADY)
#define  IS_RXNODATA(v)		((v&AERIAL_CISSTAT_MASK)==AERIAL_CISSTAT_RXNODATA)
#define  IS_TOGGLE(v,t)		((t==TOGGLE_NONE) ? 1 : (((v>>AERIAL_TOGGLE_SHIFT)&0x01)==t) )
#define  SET_TOGGLE(v,t)	{ t = ((v>>AERIAL_TOGGLE_SHIFT)&0x01)^0x01; }

#ifdef	AERIAL_FC11
static int aerial_sdio_enable_wide(struct sdio_func *func);
static int aerial_sdio_disable_wide(struct sdio_func *func);
#endif	/* AERIAL_FC11 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,31)
static int aerial_open(struct net_device *dev);
static int aerial_close(struct net_device *dev);
static int aerial_hardstart(struct sk_buff *skb, struct net_device *dev);
static int aerial_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);
struct net_device_stats * aerial_getstats(struct net_device *dev);
static void aerial_do_task(struct work_struct *work);
static int aerial_tx_thread(void *_priv);

static struct net_device_ops aerial_netdev_ops = {
         .ndo_open = aerial_open,
         .ndo_stop = aerial_close,
         .ndo_start_xmit = aerial_hardstart,
         .ndo_get_stats = aerial_getstats,
     	 .ndo_do_ioctl   = aerial_ioctl,
	    };
#endif

static int
aerial_sdio_f0_readb(struct sdio_func *func, unsigned int addr,
		     int *err_ret)
{
	int __num = func->num;
	unsigned char val;
	func->num = 0;
	val = sdio_readb(func, addr, err_ret);
	func->num = __num;
	return val;
}

static void
aerial_sdio_f0_writeb(struct sdio_func *func, unsigned char b,
		      unsigned int addr, int *err_ret)
{
	int __num = func->num;
	func->num = 0;
	sdio_writeb(func, b, addr, err_ret);
	func->num = __num;
}

int
aerial_enable_wide(struct aerial_private *priv)
{
	struct sdio_func *func = priv->func;

	if (atomic_read(&priv->power_mngt) == AERIAL_PWR_MNGT_ACTIVE) {
		return 0;
	}

	if (atomic_read(&priv->bus_type) != SDIO_FORCE_4BIT) {

#ifndef AERIAL_FC11
		sdio_enable_wide(func->card);
#else /* AERIAL_FC11 */
		aerial_sdio_enable_wide(func);
#endif /* AERIAL_FC11 */

		atomic_set(&priv->bus_type, SDIO_FORCE_4BIT);
	}
	return 0;
}

int
aerial_disable_wide(struct aerial_private *priv)
{
	struct sdio_func *func = priv->func;

	if (atomic_read(&priv->power_mngt) == AERIAL_PWR_MNGT_ACTIVE) {
		return 0;
	}

	if (atomic_read(&priv->bus_type) != SDIO_FORCE_1BIT) {
#ifndef AERIAL_FC11
		sdio_disable_wide(func->card);
#else /* AERIAL_FC11 */
		aerial_sdio_disable_wide(func);
#endif  /* AERIAL_FC11 */
		atomic_set(&priv->bus_type, SDIO_FORCE_1BIT);
	}
	return 0;
}

int
aerial_set_txcis_polling_intvl(struct aerial_private *priv, unsigned int val)
{
	if(!val){
		return -1;
	}
	priv->tx_cis_intval = val;
	return 0;
}

int
aerial_get_txcis_polling_intvl(struct aerial_private *priv, unsigned int *val)
{
	*val = priv->tx_cis_intval;
	return 0;
}

int
aerial_set_rxcis_polling_intvl(struct aerial_private *priv, unsigned int val)
{
	if(!val){
		return -1;
	}
	priv->rx_cis_intval = val;
	return 0;
}

int
aerial_get_rxcis_polling_intvl(struct aerial_private *priv, unsigned int *val)
{
	*val = priv->rx_cis_intval;
	return 0;
}

int
aerial_set_txcis_toggle_timeout(struct aerial_private *priv, unsigned int val)
{
	priv->tx_cis_toggle_timeout = val/10;
	return 0;
}

int
aerial_get_txcis_toggle_timeout(struct aerial_private *priv, unsigned int *val)
{
	*val = priv->tx_cis_toggle_timeout*10;
	return 0;
}

int
aerial_set_rxcis_toggle_timeout(struct aerial_private *priv, unsigned int val)
{
	priv->rx_cis_toggle_timeout = val/10;
	return 0;
}

int
aerial_get_rxcis_toggle_timeout(struct aerial_private *priv, unsigned int *val)
{
	*val = priv->rx_cis_toggle_timeout*10;
	return 0;
}

extern void mmc_detect_change(struct mmc_host *host, unsigned long delay);

int
aerial_restart(struct aerial_private *priv)
{
	unsigned char 	val;
	int 		ret;
	int 		timeout;
	struct sdio_func *func = priv->func;
	struct mmc_card  *card = func->card;
	struct mmc_host  *host = card->host;

	sdio_claim_host(func);

	ret = sdio_release_irq(func);
	if(ret){
		sdio_release_host(func);
		printk( KERN_ERR "failed release irq\n" );
		return -EINVAL;
	}

	val = aerial_sdio_f0_readb(func, SDIO_CCCR_ABORT, &ret);
	if (ret){
		sdio_release_host(func);
		aer_err("failed read : CIS CCCR(ABORT)\n");
		return -EIO;
	}
	
	/* reset */
	aerial_sdio_f0_writeb(func, val|0x08, SDIO_CCCR_ABORT, &ret);
	if (ret!=-ETIMEDOUT) {
		sdio_release_host(func);
		aer_err("failed reset (ret=%d)\n", ret );
		return -EIO;
	}

	sdio_release_host(func);

	mmc_detect_change(host, msecs_to_jiffies(100));
	timeout = wait_for_completion_timeout(&priv->reset_ready,
				    msecs_to_jiffies(500));
	if(!timeout){
		printk( KERN_ERR "remove timeout\n" );
		return -ETIMEDOUT;
	}

	mmc_detect_change(host, msecs_to_jiffies(100));
	return 0;
}


/* sdio-lock of this function must be taken. */
int
aerial_send_prepare(struct sdio_func *func, int size)
{
	unsigned char val;
	int retry = 100;
	int ret = -EIO;
	int flag = 0x80;

        if(size==0){
		flag=0;
	}
	/* Set block length as word(32bit) to SD2AHB register (LSB) */
	retry = 10;
	while (retry--) {
		sdio_writeb(func, size & 0xff,
			    AERIAL_F1REG_SD2AHB_BLOCKLEN_LSB, &ret);
		if (!ret) {
			val = sdio_readb(func,
					 AERIAL_F1REG_SD2AHB_BLOCKLEN_LSB,
					 &ret);
			if (val == (size & 0xff))
				break;
		}
	}
	if (ret) {
		aer_err("failed set SD2AHB register(LSB)\n");
		return -EIO;
	}

	/* Set block length as word(32bit) to SD2AHB register (MSB) */
	retry = 10;
	while (retry--) {
		sdio_writeb(func, (size >> 8) | flag,
			    AERIAL_F1REG_SD2AHB_BLOCKLEN_MSB, &ret);
		if (!ret) {
			val = sdio_readb(func,
					 AERIAL_F1REG_SD2AHB_BLOCKLEN_MSB,
					 &ret);
			if (val == ((size >> 8) | flag))
				break;
		}
	}
	if (ret) {
		aer_err("failed set SD2AHB register(MSB)\n");
		return -EIO;
	}

	return 0;
}

int
aerial_set_interrupt(struct sdio_func *func, int setint)
{
	unsigned char 	val;
	int 		ret;

	sdio_writeb(func, setint, AERIAL_F1REG_INT_MASK_REG, &ret);
	if (ret) {
		aer_err("failed set f1:int-mask\n");
		return -1;
	}
	val = sdio_readb(func, AERIAL_F1REG_INT_MASK_REG, &ret);
	if (ret || val!=setint) {
		aer_err("failed read f1:int-mask (0x%x)\n", val );
		return -1;
	}
	aer_info("TX Transmission mode %s\n", 
		(val&AERIAL_INT_TXRDY) ? "SDINT HT" : "CIS POLLING");
	aer_info("RX Transmission mode %s\n", 
		(val&AERIAL_INT_RXRCV) ? "SDINT HT" : "CIS POLLING");

	return 0;
}

/* New Code with Double Read */
int
aerial_flow_start(struct aerial_private *priv)
{
	unsigned char 	val;
	int 			ret;
	struct sdio_func *func = priv->func;

	if (atomic_read(&priv->power_mngt) != AERIAL_PWR_MNGT_ACTIVE) {
		if (aerial_send_prepare(func,AERIAL_WRITE_WORD_SIZE)) {
			aer_err("failed send prepare(TX data waiting)\n");
			return -EINVAL;
		}
	}

	while(1){
#ifdef AD500_SDIO_GPIO
		if ((sdio_gpio_enable&GPIO_TX_ENABLE) &&
			(atomic_read(&priv->state) != AERIAL_STATE_NO_FW)){
			goto GPIO_TXCIS_CHECK;
		}
#endif  /* AD500_SDIO_GPIO */

		val = aerial_sdio_f0_readb(func, AERIAL_TX_CISINFO_STATUS_ADDR, &ret);
		if (ret) {
			aer_err("failed read : CIS INFO STATUS(TX)\n");
			return -EIO;
		}

        if(!IS_TXWAIT(val))
        {
#ifdef AERIAL_SDIO_HANG_FIX
            /* In case device is not ready then release the SDIO so that RX  */
            /* interrupts are processed, if there are any                    */
            sdio_release_host(func);
            udelay(AERIAL_SDIO_TX_WAIT_DELAY);
            sdio_claim_host(func);
#endif /* AERIAL_SDIO_HANG_FIX */
            continue;
		}

#if 0
        /* Read TX status register second time in order to avoid error due to*/
        /* metastable state read                                             */
        val = aerial_sdio_f0_readb(func, AERIAL_TX_CISINFO_STATUS_ADDR, &ret);
        if (ret) {
            aer_err("failed read : CIS INFO STATUS(TX)\n");
            return -EIO;
		}
#endif /* CHECK_TX_STATUS */

        if (IS_TOGGLE(val,priv->tx_toggle)){
            SET_TOGGLE(val,priv->tx_toggle);
			break;
		}

		udelay(priv->tx_cis_intval);
        continue;

#ifdef AD500_SDIO_GPIO
GPIO_TXCIS_CHECK:

        if(!IS_IO_TXWAIT())
        {
            continue;
		}

        if (IS_IO_TXTOGGLE(priv->tx_toggle)){
            priv->tx_toggle = mxc_get_gpio_datain(AERIAL_IO_TX_TOGGLE) ^ 0x01;
			break;
		}
		udelay(priv->tx_cis_intval);
#endif  /* AD500_SDIO_GPIO */
	}

	return 0;
}

int
aerial_flow_end(struct aerial_private *priv)
{
	struct sdio_func *func = priv->func;

	if (atomic_read(&priv->power_mngt) != AERIAL_PWR_MNGT_ACTIVE) {
		if (aerial_send_prepare(func,0x00)) {
			aer_err("failed send prepare(TX data complete)\n");
			return -EINVAL;
		}
	}
	return 0;
}


static void
aerial_rx_task(struct sdio_func *func)
{
	int		ret;
	struct sk_buff *skb;
	struct aerial_private	*priv = sdio_get_drvdata(func);
	struct aerial_packet	*pr  = priv->rxbuf;
	int			size;

	ret = sdio_readsb(func, (char*)priv->rxbuf,
			  AERIAL_F1READBUFFER_ADDR, AERIAL_WRITE_SIZE);
	if (ret) {
		aer_err("failed first block read (stat=0x%x)\n", 
			sdio_readb(func, AERIAL_F1REG_FUNC1_INT_STATUS, &ret) );
		priv->net_stats.rx_errors++;
		goto RX_END;
	}

	size = pr->h.len + 2;
	if(size>AERIAL_READ_SIZE){
		aer_err("unknown size [%d]\n", size );
		priv->net_stats.rx_errors++;
		goto RX_END;
	}

	aerial_dump(LOGLVL_3, (char*)priv->rxbuf, 16);

	if (pr->h.type == HTYPE_CONFIG_RES) {
		struct aerial_wid_frame *rf = to_wid_frame(priv->rxbuf);


		if( rf->wid == WID_STATUS){
			aer_info("WID=0x%x, STATUS CODE=0x%x\n",
				rf->wid,
				 (signed char)rf->val[0] );
		}
		else
		{
			aer_debug("check_wps_msg(usbBuffQue) called l.%d\n",
				 __LINE__);
			check_wps_msg(priv, pr->message); /* usbBuffQue in windows */
        }

		if (pr->m.type == CTYPE_RESPONSE) {
			if(priv->wid_response->header)
				msleep(1);
			if(priv->wid_response->header)
				aer_err("duplicate WID response!\n");
			memcpy(priv->wid_response, priv->rxbuf, size);
			complete(&priv->wid_complete);
			return;
		}
		if (pr->m.type == CTYPE_INFO) {
			struct aerial_wid_frame *rf = to_wid_frame(priv->rxbuf);
			if (rf->wid == WID_STATUS){
				if (pr->message[7] == 0x0) {
					aer_info("disconnected!\n");
				} else {
					aer_info("connected!\n");
				}
			}
			else if (rf->wid == WID_DEVICE_READY){
				atomic_set(&priv->state, AERIAL_STATE_READY);
				aer_info("device ready!\n");
			}
			return;
		}
		if (pr->m.type == CTYPE_NETWORK) {
			struct aerial_wid_frame *rf = to_wid_frame(priv->rxbuf);
			aer_info("Network Info! [0x%04x]\n", rf->wid);
			aer_dump( &(pr->m.body[4]), (pr->m.len-8) );
		}
		return;
	}

	/* Data IN packet */
	skb = dev_alloc_skb(size + NET_IP_ALIGN);
	if (!skb) {
		priv->net_stats.rx_dropped++;
		aer_err("failed skb memory allocate error\n");
		goto RX_END;
	}
	skb->dev = priv->netdev;
	skb_reserve(skb, NET_IP_ALIGN);
	skb_put(skb, size);
	size = size - sizeof(struct aerial_packet_header);
	skb_copy_to_linear_data(skb, pr->message, size );
	skb->protocol = eth_type_trans(skb, skb->dev);

	ret = netif_rx(skb);
	if (ret == NET_RX_DROP) {
		priv->net_stats.rx_dropped++;
		aer_info("packet dropped\n");
		dev_kfree_skb_any(skb);
	}
	priv->net_stats.rx_packets++;
	priv->net_stats.rx_bytes += size;

RX_END:
	return;
}


/******************************************************************************
 * aerial_tx_task -
 *
 *****************************************************************************/
static void
aerial_tx_task(struct sk_buff *skb, struct net_device *dev)
{
	struct aerial_private *priv = netdev_priv(dev);
	struct aerial_packet txbuf;
	struct sdio_func *func = priv->func;
#ifndef	AERIAL_FC11
	unsigned long flags;
#endif /* AERIAL_FC11 */
	int ret;
	
	txbuf.h.len = skb->len;
	txbuf.h.type = HTYPE_DATA_OUT;
	memcpy(txbuf.message, skb->data, skb->len);
	dev_kfree_skb_any(skb);
	dev->trans_start = jiffies;

#ifndef	AERIAL_FC11
	local_irq_save(flags);
	if (atomic_read(&priv->power_state) == AERIAL_PWR_STATE_SLEEP) {
		local_irq_restore(flags);
		return;
	}
	sdio_claim_host(func);
	local_irq_restore(flags);
#else /* AERIAL_FC11 */
	sdio_claim_host(func);
#endif /* AERIAL_FC11 */

	aerial_led_trigger_event(priv, LED_FULL);

	if( aerial_flow_start(priv) != 0){
		goto tx_end;
	}

	ret = sdio_writesb(func, AERIAL_F1WRITEBUFFER_ADDR, &txbuf,
			   AERIAL_WRITE_SIZE);
	if (ret) {
		aer_err("failed send data\n");
		priv->net_stats.tx_errors++;
		goto tx_end;
	}
	aerial_dump(LOGLVL_3, &txbuf, 16);

	if( aerial_flow_end(priv) != 0){
		goto tx_end;
	}

	priv->net_stats.tx_packets++;
	priv->net_stats.tx_bytes += txbuf.h.len;

tx_end:
	sdio_release_host(func);

	aerial_led_trigger_event(priv, LED_OFF);
	return;
}

static void
aerial_irq(struct sdio_func *func)
{
#ifndef  RX_CIS_POLLING
	unsigned char   val;
	unsigned char   check;
	int		ret;

	/* Check Interrupt status */
	val = sdio_readb(func, AERIAL_F1REG_FUNC1_INT_STATUS, &ret);
	if (ret) {
		aer_err("failed get interrupt status\n");
		return;
	}

	if(val&(AERIAL_INT_RXRCV|AERIAL_INT_TXRDY)){
		sdio_writeb(func, val, AERIAL_F1REG_FUNC1_INT_PENDING, &ret);
		if (ret) {
			aer_err("failed clear interrupt status\n");
		}
		for (;;) {
			check = sdio_readb(func, AERIAL_F1REG_FUNC1_INT_STATUS, &ret);
			if (ret) {
				aer_err("failed get interrupt status[2]\n");
			}
			if ((check&val)==0)
				break;
			aer_err("interrupt pending? (0x%x)\n", val );
		}
	}

	if(val&AERIAL_INT_RXRCV){
		aerial_rx_task(func);
	}
#endif
	return;
}

static int
aerial_open(struct net_device *dev)
{
	struct aerial_private *priv = netdev_priv(dev);

	if (atomic_read(&priv->state) == AERIAL_STATE_NO_FW) {
		aer_err("firmware was not loadding.\n");
		return -EIO;
	}

	if (priv->bss_type!=AERIAL_BSS_WLANOFF){
		if (aerial_set_bsstype( priv, priv->bss_type)!=0){
			aer_err("failed bss type setting\n");
			return -EINVAL;
		}
	}

	netif_start_queue(dev);
	return 0;
}

static int
aerial_close(struct net_device *dev)
{
	unsigned char 		type;
	struct aerial_private 	*priv = netdev_priv(dev);

	atomic_set(&priv->state, AERIAL_STATE_READY);
	netif_stop_queue(dev);

	if (aerial_get_bsstype(priv, &type)!=0){
		aer_err("failed bss type setting\n");
		return -EINVAL;
	}
	priv->bss_type = type;
	if (priv->bss_type!=AERIAL_BSS_WLANOFF){
		if (aerial_set_bsstype( priv, AERIAL_BSS_WLANOFF)!=0){
			aer_err("failed bss type setting\n");
			return -EINVAL;
		}
	}
	return 0;
}

static int
aerial_hardstart(struct sk_buff *skb, struct net_device *dev)
{
	struct aerial_private *priv = netdev_priv(dev);
	struct aerial_task_info *task = &priv->tx_task;
	struct tx_args *args;
	unsigned long flags;

#ifndef	AERIAL_FC11
	args = kzalloc(sizeof(struct tx_args), GFP_KERNEL);
#else /* AERIAL_FC11 */
	args = kzalloc(sizeof(struct tx_args), GFP_NOWAIT);
#endif /* AERIAL_FC11 */
	if (args == NULL) {
		dev_kfree_skb_any(skb);
		aer_err("failed tx_args memory allocation\n");
		return -ENOMEM;
	}

	args->dev = dev;
	args->skb = skb;

	spin_lock_irqsave(&task->lock, flags);
	list_add_tail(&args->entry, &task->tx_args_list);
	spin_unlock_irqrestore(&task->lock, flags);

	schedule_delayed_work(&task->queue, usecs_to_jiffies(0));

	return 0;
}

static int
aerial_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	return 0;
}

struct net_device_stats *
aerial_getstats(struct net_device *dev)
{
	struct aerial_private *priv = netdev_priv(dev);

	return &priv->net_stats;
}

static void
aerial_do_task(struct work_struct *work)
{
	struct aerial_task_info *task = container_of(work,
						     struct aerial_task_info,
						     queue.work);
	struct aerial_private *priv = container_of(task,
						   struct aerial_private,
						   tx_task);

	complete(&priv->tx_thread_wait);
}

static int
aerial_tx_thread(void *_priv)
{
	struct aerial_private *priv = _priv;
	struct aerial_task_info *task = &priv->tx_task;
	struct tx_args *args;
	unsigned long flags;

	init_completion(&priv->tx_thread_wait);

	do {
		int timeout;

		timeout = wait_for_completion_interruptible_timeout(
						&priv->tx_thread_wait,
						msecs_to_jiffies(100));
		set_current_state(TASK_RUNNING);
		if (timeout > 0) {
			spin_lock_irqsave(&task->lock, flags);
			while (!list_empty(&task->tx_args_list)) {
				args = list_entry(task->tx_args_list.next,
						  struct tx_args, entry);
				list_del(&args->entry);
				spin_unlock_irqrestore(&task->lock, flags);

				aerial_tx_task(args->skb, args->dev);
				kfree(args);

				spin_lock_irqsave(&task->lock, flags);
			}
			spin_unlock_irqrestore(&task->lock, flags);
		}
	} while (!kthread_should_stop());

	return 0;
}

#ifdef RX_CIS_POLLING
static int
aerial_rx_thread(void *_priv)
{
	unsigned char 	val;
	int 			ret;
	unsigned long   start;
	long 			flags;
	struct aerial_private 	*priv = _priv;
	struct sdio_func 		*func = priv->func;

	init_completion(&priv->rx_thread_wait);

	do {
#if 0
		wait_for_completion_interruptible_timeout(
						&priv->rx_thread_wait,
						usecs_to_jiffies(priv->rx_cis_intval));
#else
		udelay(priv->rx_cis_intval);
#endif
		set_current_state(TASK_RUNNING);

		start = jiffies;
		local_irq_save(flags);
		sdio_claim_host(func);

		while(1){
#ifdef AD500_SDIO_GPIO
			if ((sdio_gpio_enable&GPIO_RX_ENABLE) &&
				(atomic_read(&priv->state) != AERIAL_STATE_NO_FW)){
				goto GPIO_RXCIS_CHECK;
			}
#endif  /* AD500_SDIO_GPIO */

			val = aerial_sdio_f0_readb(func, AERIAL_RX_CISINFO_STATUS_ADDR, &ret);
			if (ret) {
				aer_err("failed read : CIS INFO STATUS(RX)\n");
				break;
			}

			
			if (!IS_RXRDY(val)) {
				priv->rx_toggle = TOGGLE_NONE;
				break;
			}

			val = aerial_sdio_f0_readb(func, AERIAL_RX_CISINFO_STATUS_ADDR, &ret);
			if (ret) {
				aer_err("failed read : CIS INFO STATUS(RX)\n");
				break;
			}

			if (IS_TOGGLE(val,priv->rx_toggle)){
				aerial_rx_task(func);
				SET_TOGGLE(val,priv->rx_toggle);
				break;
			}
            break;

#ifdef AD500_SDIO_GPIO
GPIO_RXCIS_CHECK:

#ifndef DISABLE_GPIO_RX_STATUS
			if(!IS_IO_RXRDY())	{
				priv->rx_toggle = TOGGLE_NONE;
				break;
			}
#endif
	        if (IS_IO_RXTOGGLE(priv->rx_toggle)){
	            priv->rx_toggle = mxc_get_gpio_datain(AERIAL_IO_RX_TOGGLE) ^0x01;
				aerial_rx_task(func);
			}
			break;
#endif  /* AD500_SDIO_GPIO */
		}
		sdio_release_host(func);
		local_irq_restore(flags);

	} while (!kthread_should_stop() );

	return 0;
}
#endif

static int
aerial_task_init(struct aerial_private *priv)
{
	struct aerial_task_info *txtask = &priv->tx_task;
	int	ret;

	memset(txtask, 0, sizeof(struct aerial_task_info));

	spin_lock_init(&txtask->lock);
	INIT_LIST_HEAD(&txtask->tx_args_list);
	INIT_DELAYED_WORK(&txtask->queue, aerial_do_task);

	priv->tx_thread = kthread_run(aerial_tx_thread, priv, "%s.tx/%s",
				      AERIAL_DRVINFO, priv->netdev->name);
	if (IS_ERR(priv->tx_thread)) {
		ret = PTR_ERR(priv->tx_thread);
		aer_err("Could not create tx thread. (ret=%d)\n", ret);
		priv->tx_thread = NULL;
		return ret;
	}
#ifdef RX_CIS_POLLING
	priv->rx_thread = kthread_run(aerial_rx_thread, priv, "%s.rx/%s",
				      AERIAL_DRVINFO, priv->netdev->name);
	if (IS_ERR(priv->rx_thread)) {
		aer_err("Could not create rx thread.\n");
		return PTR_ERR(priv->rx_thread);
	}
#endif
	return 0;
}

static void
aerial_task_exit(struct aerial_private *priv)
{
	if (priv->tx_thread != NULL) {
		kthread_stop(priv->tx_thread);
		priv->tx_thread = NULL;
	}
#ifdef RX_CIS_POLLING
	if (priv->rx_thread) {
		kthread_stop(priv->rx_thread);
		priv->rx_thread = NULL;
	}
#endif
}

static int
aerial_sdio_if_init(struct sdio_func *func)
{
	unsigned char val;
	int retry;
	int ret;
	u8  setint;

	sdio_claim_host(func);

	/* set enable to Function #1 I/O */
	aerial_sdio_f0_writeb(func, 0x02, SDIO_CCCR_IOEx, &ret);
	if (ret) {
		aer_err("failed set func1 enable\n");
		goto init_end;
	}

	/* check if the I/O of Function #1 is ready */
	retry = 10;
	while (retry--) {
		val = aerial_sdio_f0_readb(func, SDIO_CCCR_IORx, &ret);
		if (ret)
			break;
		if (val & 0x02)
			break;
		msleep(10);
	}
	if (!(val & 0x02)) {
		aer_err("func1 isnot ready\n");
		goto init_end;
	}

	/* set enable Function #1 Interrupt */
	aerial_sdio_f0_writeb(func, 0x03, SDIO_CCCR_IENx, &ret);
	if (ret) {
		aer_err("failed enable interrupt\n");
		goto init_end;
	}

	sdio_enable_func(func);

	setint = 0;
#ifndef  RX_CIS_POLLING
	setint |= AERIAL_INT_RXRCV;
#endif

	if( aerial_set_interrupt(func, setint) != 0 ){
		aer_err("failed interrupt setting\n");
		goto init_end;
	}

	sdio_claim_irq(func, aerial_irq);

	/* Set block length */
	ret = sdio_set_block_size(func, AERIAL_CHUNK_SIZE);
	if (ret) {
		aer_err("failed set block size\n");
		goto init_end;
	}
#ifndef AERIAL_FC11
	sdio_enable_wide(func->card);
#else /* AERIAL_FC11 */
	aerial_sdio_enable_wide(func);
#endif /* AERIAL_FC11 */

	ret = aerial_send_prepare(func, AERIAL_WRITE_WORD_SIZE);
	if (ret) {
		aer_err("failed send prepare\n");
	}

init_end:
	sdio_release_host(func);

	return ret;
}


static struct mmc_host _default_host;

static void
aerial_set_host_configuration(struct sdio_func *func)
{
	struct mmc_card *card = func->card;
	struct mmc_host *host = card->host;
	struct mmc_host *bkup = &_default_host;

	/* backup */
	memcpy(bkup, host, sizeof(struct mmc_host));

	/* aerial specific host configruration */
	host->max_blk_size = AERIAL_CHUNK_SIZE;
}

static void
aerial_reset_host_configuration(struct sdio_func *func)
{
	struct mmc_card *card = func->card;
	struct mmc_host *host = card->host;
	struct mmc_host *bkup = &_default_host;

	/* default host configruration */
	host->max_blk_size = bkup->max_blk_size;
}

#ifdef AERIAL_FC11
static int aerial_sdio_enable_wide(struct sdio_func *func)
{
	int ret;
	u8 ctrl;
	struct mmc_card *card = func->card;
	struct mmc_host  *host = card->host;

	if (!(card->host->caps & MMC_CAP_4_BIT_DATA))
		return 0;

	if (card->cccr.low_speed && !card->cccr.wide_bus)
		return 0;

	ctrl = aerial_sdio_f0_readb(func ,SDIO_CCCR_IF, &ret);
	if (ret)
		return ret;

	ctrl |= SDIO_BUS_WIDTH_4BIT;

	aerial_sdio_f0_writeb(func, ctrl, SDIO_CCCR_IF, &ret);
	if (ret)
		return ret;
	host->ios.bus_width = MMC_BUS_WIDTH_4;
	host->ops->set_ios(host, &host->ios);
	return 0;
}

static int aerial_sdio_disable_wide(struct sdio_func *func)
{
	int ret;
	u8 ctrl;
	struct mmc_card *card = func->card;
	struct mmc_host  *host = card->host;
	
	if (!(card->host->caps & MMC_CAP_4_BIT_DATA))
		return 0;

	if (card->cccr.low_speed && !card->cccr.wide_bus)
		return 0;
	ctrl = aerial_sdio_f0_readb(func ,SDIO_CCCR_IF, &ret);
	if (ret)
		return ret;

	if (!(ctrl & SDIO_BUS_WIDTH_4BIT))
		return 0;
	ctrl &= ~SDIO_BUS_WIDTH_4BIT;

	aerial_sdio_f0_writeb(func, ctrl, SDIO_CCCR_IF, &ret);
	if (ret)
		return ret;

	host->ios.bus_width = MMC_BUS_WIDTH_1;
	host->ops->set_ios(host, &host->ios);
	return 0;
}
#endif /* AERIAL_FC11 */


static void
sdio_aerial_clean(struct aerial_private *priv)
{
	struct sdio_func *func;

	if (priv == NULL)
		return;

	/* stop threads */
	aerial_task_exit(priv);

	/* stop SDIO interrupt and function */
	func = priv->func;
	sdio_claim_host(func);
	sdio_release_irq(func);
	sdio_disable_func(func);
	sdio_release_host(func);

	if (priv->flags.sysfs_initialized)
		aerial_sysfs_remove(priv);

	if (priv->flags.netdev_registered)
		unregister_netdev(priv->netdev);

	/* free buffers */
	if (priv->fw_image	!= NULL)
		kfree(priv->fw_image);
	if (priv->rxbuf		!= NULL)
		kfree(priv->rxbuf);
	if (priv->wid_request	!= NULL)
		kfree(priv->wid_request);
	if (priv->wid_response	!= NULL)
		kfree(priv->wid_response);

	/* free netdev */
	free_netdev(priv->netdev);
}

static int
sdio_aerial_probe(struct sdio_func *func, const struct sdio_device_id *id)
{
	struct aerial_private *priv = NULL;
	struct net_device *dev;
	int ret;

#ifdef AD500_SDIO_GPIO
	mxc_set_gpio_direction(AERIAL_IO_TX_CISSTAT, 1);  /* Input  */
	mxc_set_gpio_direction(AERIAL_IO_TX_TOGGLE, 1);   /* Input  */
	mxc_set_gpio_direction(AERIAL_IO_RX_CISSTAT, 1);  /* Input  */
	mxc_set_gpio_direction(AERIAL_IO_RX_TOGGLE, 1);   /* Input  */
	
	mxc_set_gpio_dataout(AERIAL_IO_TX_CISSTAT, 0 );   /* Low	*/
	mxc_set_gpio_dataout(AERIAL_IO_TX_TOGGLE, 0 );    /* Low	*/
	mxc_set_gpio_dataout(AERIAL_IO_RX_CISSTAT, 0 );   /* Low	*/
	mxc_set_gpio_dataout(AERIAL_IO_RX_TOGGLE, 0 );    /* Low	*/
#endif /* AD500_SDIO_GPIO */

	/* net device initialize */
	dev = alloc_netdev(sizeof(struct aerial_private),
			   "rohm%d", ether_setup);
	if (!dev) {
		aer_err("could not allocate device\n");
		return -ENOMEM;
	}
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,31)
	dev->netdev_ops = &aerial_netdev_ops;
#else
	dev->open		= aerial_open;
	dev->stop		= aerial_close;
	dev->hard_start_xmit	= aerial_hardstart;
	dev->do_ioctl		= aerial_ioctl;
	dev->get_stats		= aerial_getstats;
#endif

	aerial_wireless_attach(dev);

#ifndef	AERIAL_FC11
	/* card specific capability */
	func->card->ext_caps |= MMC_CARD_CAPS_FORCE_CLK_KEEP | MMC_CARD_CAPS_FORCE_BLKXMIT;
#endif /* AERIAL_FC11 */

	/* Aerial initialize */
#ifndef	AERIAL_FC11
	priv = dev->priv;
#else /* AERIAL_FC11 */
	priv = netdev_priv(dev);
#endif /* AERIAL_FC11 */
	memset(priv, 0, sizeof(*priv));
	priv->netdev = dev;
	priv->func = func;
	priv->bss_type      = AERIAL_BSS_WLANOFF;
	priv->tx_cis_intval = AERIAL_TX_CISPOLL_INTVAL;
	priv->rx_cis_intval = AERIAL_RX_CISPOLL_INTVAL;
	priv->tx_cis_toggle_timeout = AERIAL_TX_TOGGLE_TIMEOUT/10;
	priv->rx_cis_toggle_timeout = AERIAL_RX_TOGGLE_TIMEOUT/10;
	priv->tx_toggle = TOGGLE_NONE;
	priv->rx_toggle = TOGGLE_NONE;
	
	/* allocate buffers */
	if ((priv->wid_response =
		 kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer wid_response\n");
		ret = -ENOMEM;
		goto error;
	}
	if ((priv->wid_request =
	     kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer wid_request\n");
		ret = -ENOMEM;
		goto error;
	}
	if ((priv->rxbuf =
	     kzalloc(sizeof(struct aerial_packet), GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer rxbuf\n");
		ret = -ENOMEM;
		goto error;
	}
	if ((priv->fw_image = 
	     kzalloc(FIRMWARE_SIZE_MAX, GFP_KERNEL)) == NULL) {
		aer_err("not allocate buffer fw_image\n");
		ret = -ENOMEM;
		goto error;
	}
	priv->fw_size = 0;
	
	sdio_set_drvdata(func, priv);

	atomic_set(&priv->state,	AERIAL_STATE_NO_FW);
	atomic_set(&priv->power_state,	AERIAL_PWR_STATE_ACTIVE);
	atomic_set(&priv->bus_type,	SDIO_FORCE_4BIT);
	atomic_set(&priv->power_mngt,	AERIAL_PWR_MNGT_ACTIVE);

	sema_init(&priv->wid_req_lock, 1);

	init_completion(&priv->wid_complete);
	init_completion(&priv->reset_ready);

	ret = aerial_sdio_if_init(func);
	if (ret)
		goto error;

	ret = register_netdev(dev);
	if (ret) {
		aer_err("failed register netdev\n");
		goto error;
	}
	priv->flags.netdev_registered = 1;

	ret = aerial_sysfs_init(priv);
	if (ret) {
		aer_err("failed register sysfs-entry\n");
		goto error;
	}
	priv->flags.sysfs_initialized = 1;

	ret = aerial_task_init(priv);
	if (ret)
		goto error;

	aerial_set_host_configuration(func);

	aerial_register_led_trigger(priv);

	printk(KERN_INFO "%s: registerd \"%s\" device as %s\n",
	       mmc_hostname(func->card->host), AERIAL_DRVINFO, dev->name);

	return 0;

error:
	sdio_aerial_clean(priv);
	
	return ret;
}

/******************************************************************************
 * spi_aerial_remove -
 *
 *****************************************************************************/
static void
sdio_aerial_remove(struct sdio_func *func)
{
	struct aerial_private *priv = sdio_get_drvdata(func);

	complete(&priv->reset_ready);

	aerial_unregister_led_trigger(priv);

	aerial_reset_host_configuration(func);

	sdio_aerial_clean(priv);

	return;
}

static const struct sdio_device_id sdio_aerial_ids[] = {
	{ SDIO_CLASS_WLAN, SDIO_VENDOR_ID_ROHM, SDIO_DEVICE_ID_ROHM_BU1806GU },
	{ /* end: all zeros */ },
};

MODULE_DEVICE_TABLE(sdio, sdio_aerial_ids);

static struct sdio_driver sdio_aerial_driver = {
	.probe		= sdio_aerial_probe,
	.remove		= sdio_aerial_remove,
	.name		= "sdio_aerial",
	.id_table	= sdio_aerial_ids,
};

/******************************************************************************
 * sdio_aerial_init -
 *
 *****************************************************************************/
static int __init
sdio_aerial_init(void)
{
	int ret;

	ret = sdio_register_driver(&sdio_aerial_driver);
	if (ret)
		aer_err("Registration failure.(%d)\n", ret);
	else
		aer_info("Version %s Load.\n", AERIAL_VERSION);

	return ret;
}

/******************************************************************************
 * sdio_aerial_exit -
 *
 *****************************************************************************/
static void __exit
sdio_aerial_exit(void)
{
	sdio_unregister_driver(&sdio_aerial_driver);
	aer_info("Version %s unloaded.\n", AERIAL_VERSION);
}

module_init(sdio_aerial_init);
module_exit(sdio_aerial_exit);

MODULE_AUTHOR("Nissin Systems Co.,Ltd.");
MODULE_LICENSE("GPL");
