/*
 * ebus_drv.c
 *
 * Copyright (C) 2017 Yamaha Corporation
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */
#include <asm/io.h>
#include <asm/arch/hardware.h>
#include <asm/unaligned.h>
#include <stdint.h>
#include "ebus_drv.h"

#define GPMC_BASE           0x50000000
#ifdef CONFIG_YAMAHA_EKB
#define GPMC_CS_NUM	0
#else
#define GPMC_CS_NUM	1
#endif
#define GPMC_CS1_BASE       (GPMC_BASE + GPMC_CS0_OFFSET + GPMC_CS_SIZE * GPMC_CS_NUM)
#define GPMC_CONFIG         0x50
#define GPMC_CS_SIZE        0x30
#define GPMC_CS0_OFFSET     0x60
#define GPMC_CS_CONFIG1     0x00
#define GPMC_CS_CONFIG2     0x04
#define GPMC_CS_CONFIG3     0x08
#define GPMC_CS_CONFIG4     0x0c
#define GPMC_CS_CONFIG5     0x10
#define GPMC_CS_CONFIG6     0x14
#define GPMC_CS_CONFIG7     0x18
#define GPMC_CONFIG_VAL     0x00000012
#ifdef CONFIG_YAMAHA_EKB
#define GPMC_CS_CONFIG1_VAL 0x00601100
#define GPMC_CS_CONFIG2_VAL 0x000F0F04
#define GPMC_CS_CONFIG3_VAL 0x11030302
#define GPMC_CS_CONFIG4_VAL 0x0E054E05
#define GPMC_CS_CONFIG5_VAL 0x000D1010
#define GPMC_CS_CONFIG6_VAL 0x8E050000
#define GPMC_CS_CONFIG7_VAL 0x00000f41
#else
#define GPMC_CS_CONFIG1_VAL 0x00611100
#define GPMC_CS_CONFIG2_VAL 0x000c0e04
#define GPMC_CS_CONFIG3_VAL 0x11030302
#define GPMC_CS_CONFIG4_VAL 0x0b054e05
#define GPMC_CS_CONFIG5_VAL 0x000d0c0f
#define GPMC_CS_CONFIG6_VAL 0x8b050000
#define GPMC_CS_CONFIG7_VAL 0x00000f41
#endif

#define CHIP_INITIALIZE   0xF00F
#define CLOCK_MODE_SELECT 0x3915

#define SPECIAL_KEY_MAX (4)

#if defined(CONFIG_YAMAHA_GENOS)
static struct {
	const char* name;
	EBS_DRV_SHORT_MESSAGE pattern;
}	special_sw = {
 	.name = "updatemd" ,
	.pattern = { 0x32, 0x70, 0x36, 0x00 },
};
#elif defined(CONFIG_YAMAHA_CVPPSR)
static struct {
	const char* name;
	EBS_DRV_SHORT_MESSAGE pattern;
}	special_sw = {
 	.name = "updatemd" ,
	.pattern = { 0x30, 0x70, 0x06, 0x00 },
};
#else
#error "UNKNOWN MODEL."
#endif

static const char* normal_name = "normal";
static const char* bootmode = 0;
static int ebus_ready = 0;

static void ebs_drv_WriteReg(volatile uint16_t *addr, uint8_t data)
{
	*addr = (uint16_t) data;

	return;
}

static void ebs_drv_ReadReg(volatile uint16_t *addr, uint8_t *data)
{
	*data = get_unaligned((uint32_t *) addr);

	return;
}

static void ebs_drv_ReadReg16(volatile uint16_t *addr, uint16_t *data)
{
	*data = get_unaligned((uint32_t *) addr);

	return;
}

static void ebs_drv_WaitReady(void)
{
	uint8_t opt;

	do {
		ebs_drv_ReadReg(ebs_drv_OptRegAdr, &opt);
	} while ((opt & 0x80) == 0);

	return;
}

static void ebs_drv_WaitIrq(void)
{
	uint8_t opt;

	do {
		ebs_drv_ReadReg(ebs_drv_IrqRegAdr, &opt);
	} while (opt != 0x01);

	/* clear interrupt */
	ebs_drv_WriteReg(ebs_drv_IrqRegAdr, 0x01);

	return;
}

static void ebs_drv_WaitMsecFunc(int time)
{
	mdelay(time);

	return;
}


static void ebs_drv_Wait1_3usecFunc(void)
{
	udelay(2);

	return;
}

static int ebs_drv_RecvData_timeout(EBS_DRV_SHORT_MESSAGE *msg, int timeout)
{
	uint8_t dummy;

	while (timeout) {
		ebs_drv_ReadReg(ebs_drv_IrqRegAdr, &dummy);

		if (dummy & 0x01)
			break;

		mdelay(1);
        
		if (--timeout == 0) {
			return -1;
		}
	}

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();
	ebs_drv_ReadReg(ebs_drv_DatRegAdr, &dummy);

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();
	ebs_drv_ReadReg(ebs_drv_DatRegAdr, &msg->addr);

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();
	ebs_drv_ReadReg(ebs_drv_DatRegAdr, &msg->data1);

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();
	ebs_drv_ReadReg(ebs_drv_DatRegAdr, &msg->data2);

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();
	ebs_drv_ReadReg(ebs_drv_DatRegAdr, &msg->data3);

	return 0;
}

static int dump_allRecvData(void)
{
	uint8_t status;
	EBS_DRV_SHORT_MESSAGE msg;


	if( ! ebus_ready ) return 0;

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);
	if( ! (status & EBS_DRV_BUS_AAS_BIT) ) return 0;

	while (1) {
		if (ebs_drv_RecvData_timeout(&msg, 10) != 0) {
			break;
		}
	}

	puts("RecvData Dump!!\n");
	return 0;
}

static int ebs_drv_SendData_immediate(EBS_DRV_SHORT_MESSAGE *msg)
{
	uint8_t status;

	do {
		ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);
		if( status & EBS_DRV_BUS_AAS_BIT ) dump_allRecvData();
	} while (status & EBS_DRV_BUS_BUSY_BIT);

	ebs_drv_Wait1_3usecFunc();

	do {
		ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);
		if( status & EBS_DRV_BUS_AAS_BIT ) dump_allRecvData();
	} while (status & EBS_DRV_BUS_BUSY_BIT);

	ebs_drv_WriteReg(ebs_drv_DatRegAdr, msg->addr);
	ebs_drv_WaitReady();

	ebs_drv_WriteReg(ebs_drv_StsRegAdr, 0xf8); /* start bit */
	ebs_drv_WaitReady();

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);

	ebs_drv_WriteReg(ebs_drv_DatRegAdr, 0x10); /* host */
	ebs_drv_WaitReady();

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);

	ebs_drv_WriteReg(ebs_drv_DatRegAdr, msg->data1);
	ebs_drv_WaitReady();

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);

	ebs_drv_WriteReg(ebs_drv_DatRegAdr, msg->data2);
	ebs_drv_WaitReady();

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);

	ebs_drv_WriteReg(ebs_drv_DatRegAdr, msg->data3);
	ebs_drv_WaitReady();

	ebs_drv_WaitIrq();
	ebs_drv_WaitReady();

	ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);

	ebs_drv_WriteReg(ebs_drv_StsRegAdr, 0xd8); /* stop bit */
	ebs_drv_WaitReady();

	do {
		ebs_drv_ReadReg(ebs_drv_StsRegAdr, &status);
	} while (status & EBS_DRV_BUS_MST_BIT);

	return 0;
}

static int ebs_drv_WaitInit(void)
{
	uint32_t i;
	uint16_t ini, cms;

	for( i= 0;i<10; i++){
		ebs_drv_ReadReg16(ebs_drv_IniRegAdr, &ini);
		ebs_drv_ReadReg16(ebs_drv_CmsRegAdr, &cms);
		if((ini == CHIP_INITIALIZE) && (cms == CLOCK_MODE_SELECT)) return 1;
		ebs_drv_WaitMsecFunc(100);
	}
	puts("ebs_drv_WaitInit timeout\n");

	return 0;
}

#define OP_MSG_EQUAL(a,b) ( \
		(a).addr  == (b).addr \
	&&	(a).data1  == (b).data1 \
	&&	(a).data2  == (b).data2 \
	&&	(a).data3  == (b).data3 \
	)

EBS_DRV_SHORT_MESSAGE msg_stack[SPECIAL_KEY_MAX];

static const char* analyze(void)
{
	int key_num = 0;
	int sw_num = 0;
	EBS_DRV_SHORT_MESSAGE msg = { 0, 0, 0, 0 };
	EBS_DRV_SHORT_MESSAGE key_msg[SPECIAL_KEY_MAX];
	EBS_DRV_SHORT_MESSAGE sw_msg[SPECIAL_KEY_MAX];
	int is_special = 0;

	if( ! ebus_ready ) return 0;

	while (1) {
		if (ebs_drv_RecvData_timeout(&msg, 10) != 0) {
			break;
		}

		msg.data1 &= 0xf0;
		if( (msg.addr & 0xF0 ) == EBS_CATEGORY_ID_VKS && (msg.data1 & 0xF0) == EBS_MNG_CMD_KEY_ON){
			if( key_num < SPECIAL_KEY_MAX ){
				key_msg[key_num] = msg;
			}
			key_num++;
		}
		else if( (msg.addr & 0xF0 ) == EBS_CATEGORY_ID_PNS && (msg.data1 & 0xF0) == EBS_MNG_CMD_SW_ON) {
	                printf("msg.addr  = %d\n" , msg.addr);
	                printf("msg.data1 = %d\n" , msg.data1);
	                printf("msg.data2 = %d\n" , msg.data2);
                        printf("msg.data3 = %d\n" , msg.data3);
			if( OP_MSG_EQUAL(special_sw.pattern , msg ) ){
                                printf("is_special=1\n");
				is_special = 1;
			}
			if( sw_num < SPECIAL_KEY_MAX ){
				sw_msg[sw_num] = msg;
			}
			sw_num++;
		}
	}

	if( key_num != 0 && key_num <= SPECIAL_KEY_MAX ){
		int i;
		for (i = 0; i < key_num; i++) {
			msg_stack[i] = key_msg[i];
		}
	}
	else if( sw_num != 0 && sw_num <= SPECIAL_KEY_MAX ){
		int i;
		for (i = 0; i < sw_num; i++) {
			msg_stack[i] = sw_msg[i];
		}
	}

	return is_special ? special_sw.name : normal_name;
}

static EBS_DRV_SHORT_MESSAGE* get_protcol_number(EBS_DRV_SHORT_MESSAGE *output )
{
	EBS_DRV_SHORT_MESSAGE receive = { 0, 0, 0, 0 };

	if( ! ebus_ready ) return 0;

	/* Send Protocol Number Request to Keyboard */
	{
		EBS_DRV_SHORT_MESSAGE request;
		request.addr  = 0x20; 
		request.data1 = 0x00;
		request.data2 = 0x00;
		request.data3 = 0x00;

		ebs_drv_SendData_immediate(&request);
	}

	ebs_drv_WaitMsecFunc(10);
	
	while (1) {
		if (ebs_drv_RecvData_timeout(&receive, 10) != 0) {
			 return 0;
		}

		if( (receive.addr & 0xF0 ) == EBS_CATEGORY_ID_VKS && receive.data1 == 0x00){
					*output = receive;
					return output;
		}
	}
	return 0;
}

static void ebus_gpmc_init(void)
{
	writel(GPMC_CONFIG_VAL, GPMC_BASE + GPMC_CONFIG);
	writel(GPMC_CS_CONFIG1_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG1);
	writel(GPMC_CS_CONFIG2_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG2);
	writel(GPMC_CS_CONFIG3_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG3);
	writel(GPMC_CS_CONFIG4_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG4);
	writel(GPMC_CS_CONFIG5_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG5);
	writel(GPMC_CS_CONFIG6_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG6);
	writel(GPMC_CS_CONFIG7_VAL, GPMC_CS1_BASE + GPMC_CS_CONFIG7);
}

int ebus_ShowPwmLed( uint8_t num , uint8_t onoff )
{
	uint8_t port = ( num ? 0xe1 : 0xe0 );
	uint8_t value = ( onoff ? 0xff : 0x00 );
	EBS_DRV_SHORT_MESSAGE msg =
	{ .addr = 0x32, .data1 = port, .data2 = value, .data3 = value };

	ebs_drv_SendData_immediate(&msg);

	return 0;
}

int ebus_SetLed( uint8_t dev, uint8_t num , uint8_t onoff )
{
	uint8_t addr = 0x30 | (( dev < 1 ) & 0x0E);
	uint8_t value = ( onoff ? 0xff : 0x00 );
	EBS_DRV_SHORT_MESSAGE msg =
	{ .addr = addr, .data1 = 0x70, .data2 = num, .data3 = value };

	ebs_drv_SendData_immediate(&msg);

	return 0;
}

static void turn_led_on(void)
{
	EBS_DRV_SHORT_MESSAGE msg = { 0, 0, 0, 0 };
	
	if(get_protcol_number(&msg)){
		if(msg.data2 == 0x04 && msg.data3 == 0x09){
			ebus_ShowPwmLed( 1 , 1 );
			ebus_SetLed(0,60,1);
			ebus_SetLed(0,61,1);
			ebus_SetLed(0,62,1);
			ebus_SetLed(0,63,1);
			ebus_SetLed(0,64,1);
			ebus_SetLed(0,65,1);
		}
		else if((msg.data2 == 0x08 && msg.data3 == 0x03) || (msg.data2 == 0x0A && msg.data3 == 0x13) ){
			ebus_SetLed(0,115,1);
		}
		else if((msg.data2 == 0x08 && msg.data3 == 0x06)  || ( msg.data2 == 0x0A && msg.data3 == 0x90 )) {
			ebus_SetLed(0,77,1);
		}
	}
}

int ebus_init(void)
{

	puts("GPMC:  ");

	ebus_gpmc_init();

	ebus_ready = ebs_drv_WaitInit();
	if( ! ebus_ready ){
		puts("EBus not ready\n");
		return 0;
	}

	/* Set E-Bus ICN to zero and reset devices */
	/* Hardware reset of I2C controller */
	ebs_drv_WriteReg(ebs_drv_OptRegAdr, 0x10);
	ebs_drv_WaitMsecFunc(25);
	/* Release hardware reset of I2C controller */
	ebs_drv_WriteReg(ebs_drv_OptRegAdr, 0x00);

	/* Transfer format */
	ebs_drv_WriteReg(ebs_drv_SlvRegAdr, 0x10); /* host */
	ebs_drv_WaitReady();
	ebs_drv_WriteReg(ebs_drv_CtrRegAdr, 0x10); /* 8bit, ACK, 400kHz */
	ebs_drv_WaitReady();
	ebs_drv_WriteReg(ebs_drv_StsRegAdr, 0x18); /* PIN set I2C Bus set */
	ebs_drv_WaitReady();
	ebs_drv_WriteReg(ebs_drv_IrqRegAdr, 0x00);	/* clear IRQ */

	/* set ICN to 1 */
	ebs_drv_WriteReg(ebs_drv_OptRegAdr, 0x09);

	ebs_drv_WaitMsecFunc(50);

	/* Send E-Bus start */
	{
		EBS_DRV_SHORT_MESSAGE msg;
		msg.addr  = 0x00; /* general call */
		msg.data1 = 0x01; /* E-Bus start */
		msg.data2 = 0x00; /* dummy */
		msg.data3 = 0x00; /* dummy */

		ebs_drv_SendData_immediate(&msg);
		puts("EBus ready\n");
	}

	bootmode = analyze();

	printf("bootmode = %s\n" , bootmode);

	ebs_drv_WaitMsecFunc(10);

	turn_led_on();

	/* Send Expansion parameter set */
	{
		EBS_DRV_SHORT_MESSAGE msg;
		msg.addr  = 0x32;
		msg.data1 = 0x04;
		msg.data2 = 0x00;
		msg.data3 = 0xFF;

		ebs_drv_SendData_immediate(&msg);
	}

	return 0;
}

int ebus_reset(void)
{
	/* Set ICN to 0 */
	ebs_drv_WriteReg(ebs_drv_OptRegAdr, 0x10);
	ebs_drv_WaitMsecFunc(25);
	ebs_drv_WriteReg(ebs_drv_OptRegAdr, 0x09);

	return 0;
}

const char* ebus_hotkey(void)
{
	return ( bootmode != 0 ) ? bootmode : "normal";
}

void ebus_get_hotkey(unsigned long* buf)
{
	int i;
	unsigned long temp;
	unsigned char* p = (unsigned char*)&temp;
	if( ! buf ) return;

	for( i = 0; i< SPECIAL_KEY_MAX; i++){
		temp = 0;
		p[0] = msg_stack[i].addr;
		p[1] = msg_stack[i].data2;

		*buf = temp;
		buf++;
	}
}
