/*
 * memcheck.c
 *
 * Copyright (C) 2017 Yamaha Corporation
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */
#include <stdint.h>
#include <memcheck.h>

/* ========== モデル毎変更箇所 ここから ========== */
/* RAM先頭アドレスを設定(データシートのMemory Mapを参照) */
#define TST_RAM_CHECK_START (0x80000000)

/* アドレスピン数を設定 */
#define TST_RAMCHECK_RAM_ADDR_PIN_NUM      (15) /* A0～A14 */

/* バンクアドレス数を設定 */
#define TST_RAMCHECK_RAM_BANK_ADDR_PIN_NUM (3)  /* BA0～BA2 */

#define TST_RAM_ADDRESS_SHIFT_NUM ( TST_RAMCHECK_RAM_ADDR_PIN_NUM + TST_RAMCHECK_RAM_BANK_ADDR_PIN_NUM + 1 )

/* Bit幅を設定 */
#define TST_RAM_ADDRESS_OFF_SET_32BIT (4)
#define TST_RAM_ADDRESS_OFF_SET_16BIT (2)
#define TST_RAM_ADDRESS_OFF_SET TST_RAM_ADDRESS_OFF_SET_32BIT

/* チェック用データ設定(各アドレスそれぞれユニークなデータを設定) */
static uint32_t tst_ramcheckdata[TST_RAM_ADDRESS_SHIFT_NUM] = {
    0x00000005,
    0x0000000A,
    0x00000050,
    0x000000A0,
    0x00000500,
    0x00000A00,
    0x00005000,
    0x0000A000,
    0x00050000,
    0x000A0000,
    0x00500000,
    0x00A00000,
    0x05000000,
    0x0A000000,
    0x50000000,
    0xA0000000,
    0x55555555,
    0xAAAAAAAA,
    0xFFFFFFFF,
};
/* ========== モデル毎変更箇所 ここまで ========== */

/*===================================================================================*/
/*!
 *	@brief		RAMチェック実行処理
 *
 *	@param		なし
 *
 *	@return		int            err      : 0     OK
 *	                                    : 0以外 NG
 * 
 *	@auther		st乗松
 *
 *	@data		2016/02/03
 *
*/
/*===================================================================================*/
int tst_CheckSdramCore(void)
{
    unsigned int tst_lOffset   = 0;
    unsigned int bit_shift_cnt = 0;
    unsigned int bit_shift_end = TST_RAM_ADDRESS_SHIFT_NUM;

    uint32_t    *tst_pTestAdrArr[TST_RAM_ADDRESS_SHIFT_NUM] = {};          /* 一度にチェックするアドレスの配列         */
    uint32_t    tst_lTestAdrStock[TST_RAM_ADDRESS_SHIFT_NUM] = {};          /* チェックするアドレスのDataを保存するRAM  */

    int err = 0;

    /* ADDRESS TABLE作成とDATA保存      */
    for( bit_shift_cnt = 0 ; bit_shift_cnt < bit_shift_end ; bit_shift_cnt++ ){
        /* テスト用アドレスを作成して保存 */
        tst_pTestAdrArr[bit_shift_cnt] = (uint32_t *)(TST_RAM_CHECK_START + tst_lOffset);
        /* テスト用のアドレスに格納されているデータを退避 */
        tst_lTestAdrStock[bit_shift_cnt] = *((uint32_t *)(TST_RAM_CHECK_START + tst_lOffset));

        if ( tst_lOffset == 0 ){ tst_lOffset = TST_RAM_ADDRESS_OFF_SET;   } /* 初めは4をプラス */
        else                   { tst_lOffset <<= 1; } /* 後は1bitシフトしていく */
    }

    /* テストデータを書き込む */
    for( bit_shift_cnt = 0 ; bit_shift_cnt < bit_shift_end ; bit_shift_cnt++ ){
        *tst_pTestAdrArr[bit_shift_cnt] = tst_ramcheckdata[bit_shift_cnt];
    }

    /* 読み込みチェック */
    for( bit_shift_cnt = 0 ; bit_shift_cnt < bit_shift_end ; bit_shift_cnt++ ){
        if ( *tst_pTestAdrArr[bit_shift_cnt] != tst_ramcheckdata[bit_shift_cnt] ){
            /* 書き込んだテストデータと差異があった場合はエラー */
            err++;
        }
    }

    /* 退避したデータを元に戻す */
    for( bit_shift_cnt = 0 ; bit_shift_cnt < bit_shift_end ; bit_shift_cnt++ ){
        *tst_pTestAdrArr[bit_shift_cnt] = tst_lTestAdrStock[bit_shift_cnt];
    }
    return err;
}

